# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
#
# Copyright (C) 2014 Canonical Ltd
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging

import os
import testtools
from gi.repository import Accounts
from testtools.matchers import HasLength

from uoa_integration_tests import OnlineAccounts, base


logger = logging.getLogger(__name__)


class OnlineAccountsTestCase(base.TestRunner):

    def setUp(self):
        super(OnlineAccountsTestCase, self).setUp()
        self.addCleanup(self.cleanUp)
        os.environ['AG_PROVIDERS'] = './data'
        os.environ['AG_SERVICES'] = './data'
        os.environ['SSO_LOGGING_LEVEL'] = '2'

        self.account_manager = OnlineAccounts.AccountsManager()

    def cleanUp(self):
        self.account_manager = None

    def add_account(self):
        account = self.account_manager.add_account_with_oauth_token(
                'fake-oauth', 'fake-service',
                'john@example.com', 'my secret',
                'XXX OAuth YYY')
        self.addCleanup(self.delete_account, account)
        return account

    def delete_account(self, account):
        if account.id in self.account_manager.get_accounts_list():
            self.account_manager.delete_account(account)

    def test_add_account_must_enable_it(self):
        account = self.add_account()

        self.assertTrue(account.get_enabled())

    def test_add_account_must_set_provider(self):
        account = self.add_account()

        provider_id = 'fake-oauth'
        self.assertEqual(account.get_provider_name(), provider_id)

    def test_add_account_must_enable_service(self):
        account = self.add_account()
        services = account.list_services()

        self.assertThat(services, HasLength(1))
        service_id = 'fake-service'
        self.assertEqual(services[0].get_name(), service_id)
        service = Accounts.AccountService.new(account, services[0])
        self.assertTrue(service.get_enabled())

    def test_delete_account_must_remove_it(self):
        account = self.add_account()
        self.assertThat(self.account_manager._manager.list(), HasLength(1))

        self.account_manager.delete_account(account)
        self.assertThat(self.account_manager._manager.list(), HasLength(0))

