/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtAddQuteComAccount.h"

#include "QtAddAccountUtils.h"
#include "QtLoginDialog.h"

#include "ui_AddQuteComAccount.h"

#include <presentation/qt/QtQuteCom.h>

#include <control/profile/CUserProfileHandler.h>

#include <model/account/qutecom/QuteComAccount.h>
#include <model/config/ConfigManager.h>
#include <model/config/Config.h>
#include <model/profile/UserProfile.h>
#include <model/profile/UserProfileFileStorage.h>
#include <model/profile/UserProfileHandler.h>
#include <model/webservices/url/WsUrl.h>

#include <qtutil/LanguageChangeEventFilter.h>
#include <qtutil/WidgetBackgroundImage.h>
#include <qtutil/SafeConnect.h>

#include <util/Logger.h>
#include <util/SafeDelete.h>

#include <QtGui/QtGui>

QtAddQuteComAccount::QtAddQuteComAccount(QtLoginDialog * qtLoginDialog, QWidget* parent, CUserProfileHandler & cUserProfileHandler)
	:ILogin(qtLoginDialog, cUserProfileHandler){

	_ui = new Ui::AddQuteComAccount();
	_ui->setupUi(this);

	LANGUAGE_CHANGE(this);

	WidgetBackgroundImage::setBackgroundImage(_ui->loginLabel, ":pics/headers/login.png", WidgetBackgroundImage::AdjustHeight);

	SAFE_CONNECT(_ui->loginButton, SIGNAL(clicked()), SLOT(loginClicked()));
	SAFE_CONNECT(_ui->backButton, SIGNAL(clicked()), SLOT(goBack()));
	SAFE_CONNECT_RECEIVER(_ui->cancelButton, SIGNAL(clicked()), _loginDialog, SLOT(reject()));
	SAFE_CONNECT(_ui->createQuteComAccountLabel, SIGNAL(linkActivated(const QString &)), SLOT(createAccountButtonClicked()));
	SAFE_CONNECT(_ui->helpLabel, SIGNAL(linkActivated(const QString &)), SLOT(helpButtonClicked()));
	SAFE_CONNECT(_ui->forgotPasswordLabel, SIGNAL(linkActivated(const QString &)), SLOT(forgotPasswordButtonClicked()));
}

QtAddQuteComAccount::~QtAddQuteComAccount() {
	OWSAFE_DELETE(_ui);
}

void QtAddQuteComAccount::createAccountButtonClicked() {
	WsUrl::showQuteComAccountCreation();
}

void QtAddQuteComAccount::helpButtonClicked() {
	WsUrl::showWikiPage();
}

void QtAddQuteComAccount::forgotPasswordButtonClicked() {
	WsUrl::showLostPasswordPage();
}

void QtAddQuteComAccount::loginClicked() {
	std::string login(_ui->email->text().trimmed().toUtf8());
	
	// check if login is an email address
	if (login.find("@", 0) == login.npos) {
	
		QMessageBox::warning(this, tr("@product@ - Login"),
			tr("Please enter an email address."),
			QMessageBox::Ok, QMessageBox::NoButton);
	
		return;
	}
	////
	
	std::string password( _ui->password->text().trimmed().toUtf8());
	if (!login.empty() && !password.empty()) {
		QuteComAccount qutecomAccount(login, password, true/*_ui->rememberPassword->isChecked()*/);
		qutecomAccount.setVisibleName(login);

		// Looking for the selected profile
		if (!_cUserProfileHandler.userProfileExists(login)) {
			// The selected profile does not exist. Create it.
			UserProfile userProfile;

			Config & config = ConfigManager::getInstance().getCurrentConfig();
			QtAddAccountUtils::addTestContacts(&userProfile,
				config.getQuteComAudioTestCall(),
				config.getQuteComVideoTestCall()
				);

			userProfile.setSipAccount(qutecomAccount, false /* needInitialization */);
			UserProfileFileStorage userProfileStorage(userProfile);
			userProfileStorage.save(login);
		}
		_cUserProfileHandler.setCurrentUserProfile(login, qutecomAccount);
		_loginDialog->accept();
	}
}

void QtAddQuteComAccount::setInfoMessage() {
	setLoginLabel(tr("Please enter your email address<br/>and your password"));
}

void QtAddQuteComAccount::setErrorMessage(const QString & message) {
	setLoginLabel(QString("<font color=\"red\">%1</font>").arg(message));
}

void QtAddQuteComAccount::setLoginLabel(const QString & message) {
	QString loginLabel = QString("<span style=\"font-size:16pt;\">" +
                                  tr("Login") + "</span><br/>%1").arg(message);
	_ui->loginLabel->setText(loginLabel);
}

void QtAddQuteComAccount::languageChanged() {
	_ui->retranslateUi(this);
}

void QtAddQuteComAccount::load(const SipAccount & sipAccount) {
	if (sipAccount.getType() == SipAccount::SipAccountTypeQuteCom) {
		const QuteComAccount & qutecomAccount = dynamic_cast<const QuteComAccount &>(sipAccount);
		_ui->email->setText(QString::fromUtf8(qutecomAccount.getQuteComLogin().c_str()));
		_ui->password->setText(QString::fromUtf8(qutecomAccount.getQuteComPassword().c_str()));
		setInfoMessage();
	} else {
		LOG_DEBUG("SIP getType() = SipAccount::SipAccountTypeBasic");
		//_loginDialog->changePage(QtLoginDialog::AddSIPAccount, sipAccount);
	}
}

void QtAddQuteComAccount::load(std::string sipAccountName) {
	//load userprofile
	UserProfile * userProfile = _cUserProfileHandler.getUserProfileHandler().getUserProfile(sipAccountName);
	if (userProfile && userProfile->hasSipAccount()) {
		//tests account type
		if (userProfile->hasQuteComAccount()) {
			//loads the qutecom acccount
			QuteComAccount * qutecomAccount = userProfile->getQuteComAccount();
			_ui->email->setText(QString::fromUtf8(qutecomAccount->getQuteComLogin().c_str()));
			_ui->password->setText(QString::fromUtf8(qutecomAccount->getQuteComPassword().c_str()));
			setInfoMessage();
		} else {
			//should be loaded as a sip account
			_loginDialog->changePage(QtLoginDialog::AddSIPAccount, sipAccountName);
		}
	} else {
		_loginDialog->changePage(QtLoginDialog::AddAccount);
	}
	OWSAFE_DELETE(userProfile);
}

void QtAddQuteComAccount::initPage() {
	_ui->email->setFocus();
	_ui->loginButton->setDefault(true);
	setInfoMessage();
}

void QtAddQuteComAccount::keyPressEvent(QKeyEvent *event) {
	if (event->key()==Qt::Key_Return) {
		loginClicked();
	}
}
