#!/usr/bin/env python

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#  FuzzyClock - by Igor Vatavuk

import screenlets
from screenlets.options import ColorOption , BoolOption, FontOption
from screenlets import DefaultMenuItem , sensors
import pango
import gobject
import random
import gtk
import cairo
import gconf
#use gettext for translation
import gettext

_ = screenlets.utils.get_translator(__file__)

def tdoc(obj):
	obj.__doc__ = _(obj.__doc__)
	return obj

@tdoc
class FuzzyClockScreenlet (screenlets.Screenlet):
	"""Tells time in a natural way."""
	
	# default meta-info for Screenlets (should be removed and put into metainfo)
	__name__	= 'FuzzyClock'
	__version__	= '1.1+++'
	__author__	= 'Igor Vatavuk'
	__desc__	= __doc__	# set description to docstring of class
	
	def __init__ (self, **keyword_args):
		"""Constructor"""
		#call super (width/height MUST match the size of graphics in the theme)
		screenlets.Screenlet.__init__(self, width=180, height=60, 
			uses_theme=True,ask_on_option_override=False, **keyword_args)
		
		# initialize attributes
		self.gradient = None
		self.time = ''
		
		# add Editable Options (options that are editable through the UI and automatically load and save themselves)
		self.add_options_group(_('Options'), _('Options'))
		
		self.add_option(BoolOption(_('Options'),'use_gradient', 
			True, _('Use Gradient'), _('Use a gradient for the background'), callback=self.create_gradient))
		self.add_option(ColorOption(_('Options'),'frame_color', 
			(1, 1, 1, 1), _('Frame color'), _('Frame color'), callback=self.create_gradient))
		self.add_option(FontOption(_('Options'),'font', 
			_("FreeSans"), _('Text Font'), _('Text font')))
		self.add_option(ColorOption(_('Options'),'color_text', 
			(0, 0, 0, 0.6), _('Text color'), ''))
		
		# add the default menuitems
		self.add_default_menuitems()
		
		# create the cairo gradient for the first time
		if self.use_gradient:
			self.create_gradient()

		# add a one second timer
		self.update()
		self.timer = gobject.timeout_add(20000, self.update)
	
	def create_gradient (self):
		"""Create the cairo gradient used for drawing the screenlet."""
		self.gradient = cairo.LinearGradient(0, self.height*2, 0, 0)
		self.gradient.add_color_stop_rgba(1,*self.frame_color)
		self.gradient.add_color_stop_rgba(0.7, self.frame_color[0], self.frame_color[1],
			self.frame_color[2], 1.5-self.frame_color[3])
	
	def update (self):
		"""Timeout called every second to update the time and redraw the screenlet."""
		h = int(sensors.cal_get_hour24())
		Hours = int(sensors.cal_get_hour12())

		if h < 12:
			TimeOfDay = _(" in the morning")
		if h >= 12:
			TimeOfDay = _(" in the afternoon")
		if h >= 18:
			TimeOfDay = _(" in the evening")

		FuzzyMinute = int(sensors.cal_get_minute())

		if (FuzzyMinute > 30):
			Hours += 1

		if (FuzzyMinute <= 30):
			TextTime = _(" past ")
		if (FuzzyMinute > 30):
			TextTime = _(" to ")
		if (FuzzyMinute < 5):
			TextTime = _(" o\'clock")

		TextMins = [_('five'), _('ten'), _('quarter'), _('twenty'), _('twenty five'), _('half'), _('twenty five'), _('twenty'), _('quarter'), _('ten'), _('five')]
		TextMin = TextMins[int(FuzzyMinute / 5 -1)]
		TextHours = [_('one'), _('two'), _('three'), _('four'), _('five'), _('six'), _('seven'), _('eight'), _('nine'), _('ten'), _('eleven'), _('twelve')]
		TextHour = TextHours[Hours-1]

		if FuzzyMinute < 5:
			self.time = TextHour + TextTime + TimeOfDay
		else:
			self.time = TextMin + TextTime + TextHour + TimeOfDay

		self.redraw_canvas()
		return True # keep on running this event
	
	def on_draw (self, ctx):
		"""Called automatically to (re)draw the screenlet."""
		ctx.scale(self.scale, self.scale)
		
		if self.use_gradient and self.gradient is not None:
			ctx.set_source(self.gradient)
		else:
			ctx.set_source_rgba(*self.frame_color)
			
		self.draw_rectangle_advanced (ctx, 0, 0, self.width-12, self.height-12,
			rounded_angles=(5,5,5,5), fill=True, border_size=2, border_color=(0,0,0,0.5),
			shadow_size=6, shadow_color=(0,0,0,0.5))

		ctx.set_source_rgba(*self.color_text)

		self.draw_text(ctx, self.time, 9, 15, self.font.split(' ')[0], 11, self.width-17, pango.ALIGN_CENTER)	

	def on_draw_shape (self, ctx):
		self.draw_rectangle_advanced (ctx, 0, 0, self.width-12, self.height-12,
			rounded_angles=(5,5,5,5), fill=True, border_size=2, border_color=(0,0,0,0.5),
			shadow_size=6, shadow_color=(0,0,0,0.5))
	
# If the program is run directly or passed as an argument to the python
# interpreter then create a Screenlet instance and show it
if __name__ == "__main__":
	# create new session
	import screenlets.session
	screenlets.session.create_session(FuzzyClockScreenlet)

