//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sim/Simulation/OffspecSimulation.h
//! @brief     Defines class OffspecSimulation.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_SIM_SIMULATION_OFFSPECSIMULATION_H
#define BORNAGAIN_SIM_SIMULATION_OFFSPECSIMULATION_H

#include "Base/Types/OwningVector.h"
#include "Sim/Simulation/ISimulation.h"

class Datafield;
class IBeamScan;
class IPixel;
class OffspecDetector;

//! Off-specular scattering simulation.
//!
//! Holds an instrument and sample model.
//! Computes reflected and scattered intensity as function of incident and final glancing angle.

class OffspecSimulation : public ISimulation {
public:
    OffspecSimulation(const IBeamScan& scan, const MultiLayer& sample,
                      const OffspecDetector& detector);
    ~OffspecSimulation() override;

    std::string className() const final { return "OffspecSimulation"; }

#ifndef SWIG
    const ICoordSystem* simCoordSystem() const override;

    std::vector<const INode*> nodeChildren() const override;

    const IBeamScan* scan() const
    {
        return m_scan.get();
    }
    const OffspecDetector& detector() const
    {
        return *m_detector;
    }

private:
    //... Overridden executors:
    void initDistributionHandler() override;

    void prepareSimulation() override;

    void runComputation(const ReSample& re_sample, size_t iElement, double weight) override;

    //... Overridden getters:
    bool force_polarized() const override;

    size_t nElements() const override;

    SimulationResult packResult() override;

    //... Model components:
    std::unique_ptr<IBeamScan> m_scan;
    std::unique_ptr<OffspecDetector> m_detector;

    //... Caches:
    OwningVector<const IPixel> m_pixels; //!< All unmasked pixels inside ROI.
#endif                                   // SWIG
};

#endif // BORNAGAIN_SIM_SIMULATION_OFFSPECSIMULATION_H
