#include "includes.h"

/*
** System Activity Report
**
** The program 'atsar' offers a number of options to report about
** the statistics of important system-resources.
** The program has been designed to operate as a framework to which
** new options can be added very easily (table-driven). It cooperates
** with the data-collector program 'atsadc' in a similar way as the
** standard 'sar' operates with 'sadc'.
** ================================================================
** Author:      Gerlof Langeveld - AT Computing, Nijmegen, Holland
** E-mail:      gerlof@ATComputing.nl
** Date:        Januar  1995
** LINUX-port:  Februar 1999
** 
** $Log: atsar.c,v $
** Revision 1.9  2004/07/08 12:52:38  gerlof
** Adapted for kernel version 2.6.
**
** Revision 1.8  2001/03/14 11:09:44  gerlof
** Minor bug-fix.
**
** Revision 1.7  2001/03/12 14:56:24  gerlof
** Support flag -S to get time-stamp for every counter-line.
**
** Revision 1.6  2000/11/07 09:21:58  gerlof
** Minor bug-fixes.
**
** Revision 1.5  2000/04/25 08:48:28  gerlof
** Disable stdout-buffering to enable output-processing via pipes.
**
** Revision 1.4  1999/09/16 09:25:57  gerlof
** Minor changes for port to Alpha-platform.
**
** Revision 1.3  1999/08/26 06:53:02  gerlof
** Code-cleanup and adding new parameter when calling printline-function.
**
** Revision 1.2  1999/05/18 08:30:34  gerlof
** Back-port from version 2.2 to 2.0.
**
** Revision 1.1  1999/05/05 11:40:03  gerlof
** Initial revision
**
**
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public License as published by the
** Free Software Foundation; either version 2, or (at your option) any
** later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU General Public License for more details.
*/

static const char rcsid[] = "$Id: atsar.c,v 1.9 2004/07/08 12:52:38 gerlof Exp $";

#define	EQUAL	0
#define	SECSDAY	86400

/*
** definition table for all flags and corresponding functions
*/
extern struct funcdef	funcdef[];
extern int		funcnt;

/*
** values of the command-line arguments
*/
char		*infile;	/* input file-name       */
int		ifd;		/* input file-descriptor */

char		*interval;	/* argument: number-of-seconds interval */
char		*nsamples;	/* argument: number-of-intervals 	*/
char		numsamp[32];	/* nsamples+1 as argument for atsadc	*/
int		nflags;		/* number of options required		*/
int		newinterval;	/* argument: overruling interval with -i*/
int		daynumber;	/* day-number specified i.s.o. full name*/
char		Sflag;		/* every output-line needs time-stamp?  */

extern char	*optarg;
extern int	optind, opterr;

/*
** list of counters which are available
*/
struct filehdr	filehdr;
struct osrel	osrel;

struct countdef	*oncelist;		/* read-once counter definition  */
char		*ponce, *conce;		/* read-once counter values      */

struct countdef	*samplist;		/* per-sample counter definition */
char		*psamp, *csamp;		/* per-sample counter values     */

/*
** function prototypes
*/
int	readset(struct samphdr *, void *);
int	readin(int, char *, int);
void	prusage(char *);
void	printhead(struct utsname *, time_t);
void	mktim(time_t, char *);
void	mkdat(time_t, char *);
int	convtim(char *, unsigned int *);
int	daysecs(time_t);


int
main(int argc, char *argv[])
{
	register int	i;
	int		c;
	unsigned int	stoffset, offset;
	char		asbuf[100];
	char		*flaglist;
	unsigned int	starttime = 0, endtime=SECSDAY-1;

	struct samphdr	*phead, *chead;
	char		*oncenames, *sampnames;

	/*
	** prepare verification of command-line options
	*/
	if ( (flaglist = (char *) malloc(funcnt+12)) == NULL)
	{
		perror("malloc");
		exit(5);
	}

	for (i=0; i < funcnt; i++) 		/* gather function flags */
		*(flaglist+i) = funcdef[i].flag;

	*(flaglist+i) = 0;			/* set terminator   */

	strcat(flaglist, "As:e:i:f:n:S");	/* add standard flags */

	/*
	** verify if a flag has been used twice in this list (programming
	** error)
	*/
	for (c=strlen(flaglist)-1; c; c--)
	{
		if ( *(flaglist+c) == ':' )
			continue;

		for (i=0; i<c; i++)
		{
			if ( *(flaglist+c) != *(flaglist+i) )
				continue;

			/* ambiguous flag found! */
			fprintf(stderr,
				"programming-failure: ambiguous flag '%c'\n\n",
				*(flaglist+c));
			prusage(argv[0]);
		}
	}

	/*
	** check which command-line options are defined
	*/
	while ( (c = getopt(argc, argv, flaglist)) != EOF)
	{
		switch (c)
		{
		   case 'A':
			for (i=0; i < funcnt; i++)
				funcdef[i].active = 1;
			break;

		   case 'f':		/* input file definition */
			infile = optarg;
			break;

		   case 's':		/* start time definition */
			if ( !convtim(optarg, &starttime) )
				prusage(argv[0]);
			break;

		   case 'e':		/* end   time definition */
			if ( !convtim(optarg, &endtime) )
				prusage(argv[0]);
			break;

		   case 'i':		/* overruling interval   */
			if ( !isdigit(*optarg) )
				prusage(argv[0]);

			newinterval = atoi(optarg);
			break;

                   case 'n':
                        if ( !isdigit(*optarg) )
                                prusage(argv[0]);

                        daynumber = atoi(optarg);
                        break;

		   case 'S':		/* stamp used for every output-line */
			Sflag = 1;
			break;

		   default:
			/*
			** check which function-flag is used and mark it 
			** active
			*/
			for (i=0; i < funcnt; i++)
			{
				if (c == funcdef[i].flag)
				{
					funcdef[i].active = 1;
					break;
				}
			}

			if (i == funcnt)	/* non-existing flag ? */
				prusage(argv[0]);
		}
	}

	/*
	** if no flag has been specified, take first function as default
	*/
	for (i=0, nflags=0; i < funcnt; i++)
	{
		if (funcdef[i].active)
			nflags++;
	}

	if (!nflags)			/* no active function found ? */
	{
		funcdef[0].active = 1;
		nflags = 1;
	}

	/*
	** consider the remaining arguments (if any) as interval 
	** and number of samples
	*/
	if ( (argc - optind) > 0 )
	{
		if ( !isdigit(*argv[optind])  )
			prusage(argv[0]);

		interval = argv[optind++];

		if ( atoi(interval) <= 0 )
			prusage(argv[0]);

		if ( (argc - optind) > 0 )	/* still more args ? */
		{
			if ( !isdigit(*argv[optind])  )
				prusage(argv[0]);

			nsamples = argv[optind];
		}
		else
			nsamples = "1";		/* set default to 1 sample */

		/* add one to samples for the data-collector 'atsadc' */
		i = atoi(nsamples);
		i++;
		sprintf(numsamp, "%d", i);

		nsamples = numsamp;
	}

	/*
	** undo buffering for stdout to enable processing of the output
	** via e.g. a pipe
	*/
	setbuf(stdout, NULL);

	/*
	** open the counter input file
	*/
	if (!infile)
	{
		/*
		** no input file defined; check if on-the-fly statistics
		** wanted or statistics from default file (= statistics
		** of today untill this very moment)
		*/
		if (interval == NULL && nsamples == NULL)
		{
			struct tm 	*tt;
			time_t		curtime;

			if (daynumber)  /* specific day wanted */
			{
				sprintf(asbuf, ATOUTF, daynumber);
			}
                        else
                        {               /* today wanted        */
				curtime = time(0);
				tt = localtime(&curtime);

				sprintf(asbuf, ATOUTF, tt->tm_mday);
			}

			if ( (ifd = open(asbuf, O_RDONLY)) == -1)
			{
				fprintf(stderr, "open %s: ", asbuf);
				perror("");
				exit(1);
			}
		}
		else
		{
			int	fds[2];

			/*
			** on-the-fly analysis; activate backend 
			** data-collector
			*/
			if ( pipe(fds) == -1 )
			{
				perror("open pipe");
				exit(1);
			}

			switch ( fork() )
			{
			   case 0:		/* child */
				(void) close(1);
				(void) dup(fds[1]);
				(void) close(fds[0]);
				(void) close(fds[1]);

				execlp(ATEXEC, "atsadc",
						 interval, nsamples, 0);
				perror("exec atsadc");
				exit(1);

			   case -1:
				perror("fork");
				exit(1);

			   default:		/* parent */
				ifd = fds[0];
				(void) close(fds[1]);
			}
		}
	}
	else
	{
		/*
		** input-file specified on command-line: open it
		*/
		if ( (ifd = open(infile, O_RDONLY)) ==-1)
		{
			perror("open input-file");
			exit(1);
		}
	}

	/*
	** input-file (or pipe) is opened; start reading ......
	**
	** first read the general file-header and check if
	** it really concerns a statistics file
	*/
	if ( readin(ifd, (char *)&filehdr, sizeof(filehdr)) == 0 )
		exit(0);

	if (filehdr.magic != ATMAGIC)
	{
		fprintf(stderr, 
			"Not a proper statistics file\n");
		exit(6);
	}

	if (filehdr.hdrsize != sizeof(struct filehdr))
	{
		fprintf(stderr, 
			"Statistics file with incompatible format\n");
		exit(6);
	}

	/*
	** determine kernel-version of measured system
	*/
        sscanf(filehdr.utsinfo.release, "%d.%d.%d",
                &(osrel.rel), &(osrel.vers), &(osrel.sub));

	/*
	** read list of variable names for once-counters
	*/
	if ( (oncelist = (struct countdef *) 
			malloc(filehdr.oncecnt*sizeof(struct countdef)))==NULL)
	{
		perror("malloc");
		exit(5);
	}

	if ( readin(ifd, (char *)oncelist,
	            filehdr.oncecnt*sizeof(struct countdef))==0)
		exit(0);

	/*
	** read list of names for once-counters
	*/
	if ( (oncenames = (char *) malloc(filehdr.oncenames)) == NULL)
	{
		perror("malloc");
		exit(5);
	}

	if ( readin(ifd, (char *)oncenames, filehdr.oncenames) == 0)
		exit(0);

	/*
	** correct offsets for names in counter-list to real pointers
	** change counter-sizes into offsets for counters
	*/
	for (i=0, offset=0; i < filehdr.oncecnt; i++)
	{
		int	soffset;

		soffset  = offset;
		offset	+= (oncelist+i)->ksize * (oncelist+i)->kinst;

		(oncelist+i)->kname += (unsigned long) oncenames;
		(oncelist+i)->ksize  = soffset;
	}

	/*
	** read list of variable names for samp-counters
	*/
	if ( (samplist = (struct countdef *) 
			malloc(filehdr.sampcnt*sizeof(struct countdef)))==NULL)
	{
		perror("malloc");
		exit(5);
	}

	if ( readin(ifd, (char *)samplist,
	            filehdr.sampcnt*sizeof(struct countdef)) ==0)
		exit(0);

	/*
	** read list of names for samp-counters
	*/
	if ( (sampnames = (char *) malloc(filehdr.sampnames)) == NULL)
	{
		perror("malloc");
		exit(5);
	}

	if ( readin(ifd, (char *)sampnames, filehdr.sampnames) == 0)
		exit(0);

	/*
	** correct offsets for names in counter-list to real pointers
	** change counter-sizes into offsets for counters
	*/
	for (i=0, offset=0; i < filehdr.sampcnt; i++)
	{
		int	soffset;

		soffset  = offset;
		offset	+= (samplist+i)->ksize * (samplist+i)->kinst;

		(samplist+i)->kname += (unsigned long) sampnames;
		(samplist+i)->ksize  = soffset;
	}

	/*
	** read the once-counter values into 'current' and take care
	** that the 'previous' counters are zero
	*/
	if ( (ponce = (char *) malloc(filehdr.oncevars)) == NULL)
	{
		perror("malloc");
		exit(5);
	}

	memset(ponce, 0, filehdr.oncevars);

	if ( (conce = (char *) malloc(filehdr.oncevars)) == NULL)
	{
		perror("malloc");
		exit(5);
	}

	if ( readin(ifd, (char *)conce, filehdr.oncevars) == 0)
		exit(0);

	/*
	** create two areas for the sample-counters and header:
	** one to hold the previous sample and one to hold the current;
	** the previous sample will be filled with the start-counters
	*/
	if ( (phead = (struct samphdr *) malloc(sizeof(struct samphdr)))==NULL)
	{
		perror("malloc");
		exit(5);
	}

	if ( (chead = (struct samphdr *) malloc(sizeof(struct samphdr)))==NULL)
	{
		perror("malloc");
		exit(5);
	}

	if ( (psamp = (char *) malloc(filehdr.sampvars)) == NULL)
	{
		perror("malloc");
		exit(5);
	}

	if ( (csamp = (char *) malloc(filehdr.sampvars)) == NULL)
	{
		perror("malloc");
		exit(5);
	}

	/*
	** INPUT FROM PIPE
	*/
	if (!infile && interval != NULL)
	{
		char timbuf[16], ptimbuf[16];

		/*
		** read the initial sample header and counters
		*/
		if ( !readset(phead, psamp) )
			exit(0);

		/*
		** print header info
		*/
		printhead(&filehdr.utsinfo, phead->curtim);

		/*
		** print header line if only one flag required
		*/
		if (nflags == 1)
		{
			mktim(phead->curtim, ptimbuf);

			for (i=0; i < funcnt; i++)
			{
				if (funcdef[i].active)
				{
					printf("%s  ", ptimbuf);
					(funcdef[i].prihead)(&osrel);
					break;
				}
			}
		}

		/*
		** print sample-sets untill EOF
		*/
		while ( readset(chead, csamp) )
		{
			time_t		deltasec, deltatic;
			char		valid;
			char		*hsamp;
			struct samphdr	*hhead;

			/*
			** calculate the interval of the last sample
			*/
			deltasec	= chead->curtim   - phead->curtim;
			deltatic	= chead->curlbolt - phead->curlbolt;

			if (!deltasec)
				deltasec = 1;	/* be sure: no divide 0 */

			/*
			** print all counter lines
			*/
			mktim(chead->curtim, timbuf);
			mktim(phead->curtim, ptimbuf);

			for (i=0; i < funcnt; i++)
			{
				if (! funcdef[i].active)
					continue;

				if (nflags > 1)
				{
					printf("\n%s  ", ptimbuf);
					(funcdef[i].prihead)(&osrel);
				}

				printf("%s  ", timbuf);

				valid = (funcdef[i].priline) (
						deltasec, deltatic,
						filehdr.hertz, filehdr.numcpu,
						&osrel,
						Sflag ? timbuf : "        ");

				if ( !valid )	/* no counters */
				{
					printf("     not supported\n");
					funcdef[i].active = 0;
				}
			}

			if (nflags > 1)
				printf("\n");

			/*
			** make the current counters 'previous'
			*/
			hhead = chead;
			chead = phead;
			phead = hhead;

			hsamp = csamp;
			csamp = psamp;
			psamp = hsamp;
		}

		(void) close(ifd);

		exit(0);
	}

	/*
	** INPUT FROM FILE
	**
	** read the counter-sets untill the correct start-time is found
	** and remember the start-offset to be able to make
	** several passes (one per flag) because this is a file and NO pipe
	*/
	do
	{
		stoffset = lseek(ifd, 0, SEEK_CUR);

		if ( !readset(phead, psamp) )
			exit(0);
	} while ( starttime > daysecs(phead->curtim) );

	/*
	** print header line of total report containing the date and
	** system-info
	*/
	printhead(&filehdr.utsinfo, phead->curtim);

	/*
	** handle every defined flag
	*/
	for (i=0; i < funcnt; i++)
	{
		if (funcdef[i].active)
		{
			char	timbuf[16], valid;

			/*
			** rewind input file
			*/
			(void) lseek(ifd, stoffset, SEEK_SET);

			/*
			** read initial sample header and counters
			*/
			if ( !readset(phead, psamp) )
				exit(0);

			if (phead->cntsize == 0)
				memset(psamp, 0, filehdr.sampvars);

			/*
			** print header line
			*/
			mktim(phead->curtim, timbuf);
			printf("%s  ", timbuf);

			(funcdef[i].prihead)(&osrel);

			/*
			** print sample-sets untill EOF
			*/
			while ( readset(chead, csamp) )
			{
				time_t 		deltasec, deltatic;
				char		*hsamp;
				struct samphdr	*hhead;

				/*
				** check if specified endtime exceeded
				*/
				if (endtime < daysecs(chead->curtim) )
					break;

				mktim(chead->curtim, timbuf);

				/*
				** if special reboot record found,
				** print a restart line
				*/
				if (chead->cntsize == 0)
				{
					printf("%s     linux restarts\n", 
								timbuf);

					hhead = chead;
					chead = phead;
					phead = hhead;

					memset(psamp, 0, filehdr.sampvars);
					continue;
				}

				/*
				** check if overruling interval time is
				** already exceeded (as specified with 
				** the -i flag)
				*/
				if (newinterval)
				{
					if ( (chead->curtim+2) < 
						   (phead->curtim+newinterval) )
						continue;
				}
	
				/*
				** calculate the interval of the last sample
				*/
				deltasec = chead->curtim   - phead->curtim;
				deltatic = chead->curlbolt - phead->curlbolt;
	
				if (!deltasec)
					deltasec = 1;	/* no divide 0 */

				/*
				** print counter line
				*/
				printf("%s  ", timbuf);

				valid = (funcdef[i].priline) (
						deltasec, deltatic,
						filehdr.hertz, filehdr.numcpu,
						&osrel,
						Sflag ? timbuf : "        ");

				if ( !valid )	/* cannot find counters */
				{
					printf("     not supported\n");
					funcdef[i].active = 0;
					break;
				}

				/*
				** make the current counters 'previous'
				*/
				hhead = chead;
				chead = phead;
				phead = hhead;
	
				hsamp = csamp;
				csamp = psamp;
				psamp = hsamp;
			}

			printf("\n");
		}
	}

	(void) close(ifd);

	exit(0);
}

/*
** Read one set of sample header-data and counter-data.
**
** Return-value:	0 - End-Of-File
**			1 - Success
*/
int
readset(struct samphdr *hdr, void *cnt)
{
	/*
	** read header info
	*/
	if ( readin(ifd, (char *)hdr, sizeof(struct samphdr)) == 0) 
		return(0); 				/* EOF */

	if (hdr->cntsize == 0)
		return(1);

	/*
	** read counter contents
	*/
	if ( readin(ifd, (char *)cnt, hdr->cntsize) == 0)
		return(0);				/* EOF */

	return(1);
}

/*
** Read the input-file:
** This is a separate function because the input-file can be a pipe
** and it is likely that the blocks which are transferred exceed the
** atomicity-factor of the target system; this means that a block which
** is written as e.g. 6 Kbytes may arrive in pieces and has to be reassembled
** here.
**
** Return-value:	0 - End-Of-File
**			1 - Success
*/
int
readin(int fd, char *buf, int sz)
{
	register int n;

	while (sz)	/* as long as the requested buffer is not filled */
	{
		if ( (n = read(fd, buf, sz)) == -1 )
		{
			perror("read input-file");
			exit(1);
		}

		if (n == 0)
			return(0);		/* EOF reached */

		sz	-= n;
		buf	+= n;
	}
	return(1);
}

/*
** Print the usage-string and terminate abnormally.
*/
void
prusage(char *pname)
{
	register int	i;

	fprintf(stderr, 
		"usage: %s [-flags] t [n]              or\n", pname);
	fprintf(stderr, 
		"       %s [-flags] [-s hh:mm] [-e hh:mm] [-i sec] "
		"[-n day# | -f file]\n", pname);

	fprintf(stderr, "flags:\n");
	fprintf(stderr, "\t-A\tall flags\n");
	fprintf(stderr, "\t-S\ttime-stamp for every output-line\n");

	for (i=0; i < funcnt; i++)
	{
		fprintf(stderr, "\t-%c\t%s", 
				funcdef[i].flag, funcdef[i].about);

		if (!i)		/* first entry ? */
			fprintf(stderr, " (default flag)\n");
		else
			fprintf(stderr, "\n");
	}

	exit(1);
}

/*
** Print header-information about measured machine and
** the measurement-date
*/
void
printhead(struct utsname *uname, time_t mtime)
{
	char		datstring[16];

	mkdat(mtime, datstring);

	printf("\n%s  %s  %s  %s  %s  %s\n\n",
		uname->sysname,
		uname->nodename,
		uname->release,
		uname->version,
		uname->machine,
		datstring);
}

/*
** Generate ascii time in format hh:mm:ss
*/
void
mktim(time_t itime, char *otime)
{
	struct tm *tt;

	tt = localtime(&itime);

	sprintf(otime, "%02d:%02d:%02d", tt->tm_hour, tt->tm_min, tt->tm_sec);
}

/*
** Generate ascii date in format mm/dd/yyyy
*/
void
mkdat(time_t idate, char *odate)
{
	struct tm *tt;

	tt = localtime(&idate);

	sprintf(odate, "%02d/%02d/%04d", 
				tt->tm_mon+1, tt->tm_mday, tt->tm_year+1900);
}

/*
** Convert a hh:mm string into a number of seconds since 00:00
**
** Return-value:	0 - Wrong input-format
**			1 - Success
*/
int
convtim(char *itim, unsigned int *otim)
{
	register int	i;
	int		hours, minutes;

	/*
	** check string syntax
	*/
	for (i=0; *(itim+i) != 0; i++)
		if ( !isdigit(*(itim+i)) && *(itim+i) != ':' )
			return(0);

	sscanf(itim, "%d:%d", &hours, &minutes);

	if ( hours < 0 || hours > 24 || minutes < 0 || minutes > 60 )
		return(0);

	*otim = (hours * 3600) + (minutes * 60);

	if ( *otim >= SECSDAY)
		*otim = SECSDAY-1;

	return(1);
}

/*
** Return number of seconds since midnight according local clock time
**
** Return-value:	Number of seconds
*/
int
daysecs(time_t itime)
{
	struct tm *tt;

	tt = localtime(&itime);

	return( (tt->tm_hour*3600) + (tt->tm_min*60) );
}


/*
** Get counterset 
** ==============
** To be called by the print-line functions to obtain a pointer
** to the current counter-values and a pointer to the previous counter-values.
**
** Input:	name of the counter(struct) which is required
**		location to store 'current'  pointer
**		location to store 'previous' pointer
**
** Output:	'current' pointer
** 		'previous' pointer
**
** Returns:	number of counter-instances 
**		this implies that 0 instances is FALSE (so not found)
**
** The given name is always searched in the per-sample counters first.
*/
int
getcset(char *name, void **curptr, void **preptr)
{
	register int 	i;
	static	 int	lastused;	/* assume that only one set used */

	/*
	** check if the last used name is requested again (performance optim)
	*/
	if ( strcmp(name, (samplist+lastused)->kname) == EQUAL )
	{
		*curptr = csamp + (samplist+lastused)->ksize;
		*preptr = psamp + (samplist+lastused)->ksize;
		return( (samplist+lastused)->kinst);
	}

	/*
	** linear search through per-sample list
	** note: if the list of names in future will be very long,
	**       it might be interesting to implement some hash-list here ....
	**       however profiling has proven that only 1% of the CPU-time 
	**       consumption of a total 'atsar -A' run is spent in the
	**       'getcset' function, which is not worth to implement hashing
	*/
	for (i=0; i < filehdr.sampcnt; i++)
	{
		if ( strcmp(name, (samplist+i)->kname) == EQUAL )
		{
			*curptr = csamp + (samplist+i)->ksize;
			*preptr = psamp + (samplist+i)->ksize;

			lastused= i;
			return((samplist+i)->kinst);
		}
	}

	/*
	** linear search through read-once list
	** note: if the list of names in future will be very long,
	**       it might be interesting to implement some hash-list here...
	**       however profiling has proven that only 1% of the CPU-time 
	**       consumption of a total 'atsar -A' run is spent in the
	**       'getcset' function, which is not worth implementing hashing
	*/
	for (i=0; i < filehdr.oncecnt; i++)
	{
		if ( strcmp(name, (oncelist+i)->kname) == EQUAL )
		{
			*curptr = conce + (oncelist+i)->ksize;
			*preptr = ponce;
			return((oncelist+i)->kinst);
		}
	}

	return(0);	/* not found at all */
}
