------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--                  G N A T C H E C K . D I A G N O S E S                   --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                     Copyright (C) 2005-2010, AdaCore                     --
--                                                                          --
-- GNATCHECK  is  free  software;  you can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software Foundation;  either version 2, or ( at your option)  any  later --
-- version.  GNATCHECK  is  distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General Public License distributed with GNAT; see file  COPYING. If --
-- not,  write to the  Free Software Foundation,  51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
--                                                                          --
-- GNATCHECK is maintained by AdaCore (http://www.adacore.com).             --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Calendar;               use Ada.Calendar;
with Ada.Characters.Handling;
with Ada.Command_Line;
with Ada.Containers.Ordered_Sets;
with Ada.Exceptions;
with Ada.Strings;                use Ada.Strings;
with Ada.Strings.Fixed;          use Ada.Strings.Fixed;
with Ada.Text_IO;                use Ada.Text_IO;

with GNAT.OS_Lib;                use GNAT.OS_Lib;

with Atree;                      use Atree;
with Gnatvsn;                    use Gnatvsn;
with Sinput;                     use Sinput;
with Table;
with Types;

with Asis.Set_Get;               use Asis.Set_Get;

with ASIS_UL.Common;             use ASIS_UL.Common;
with ASIS_UL.Compiler_Options;
with ASIS_UL.Misc;               use ASIS_UL.Misc;
with ASIS_UL.Options;            use ASIS_UL.Options;
with ASIS_UL.Output;             use ASIS_UL.Output;

with Gnatcheck.Exemption;        use Gnatcheck.Exemption;
with Gnatcheck.Options;          use Gnatcheck.Options;
with Gnatcheck.Rules.Output;     use Gnatcheck.Rules.Output;
with Gnatcheck.Rules.Rule_Table; use Gnatcheck.Rules.Rule_Table;

package body Gnatcheck.Diagnoses is

   --  Because of histiorical reasons, we have two different formats of the
   --  report file and the corresponding mechanisms. One (referenced below as
   --  old format) is rather verbose and can report rule violations oredered
   --  in different ways. This format contains neither compiler error messages
   --  for non-compilable files nor warnings from rule exemption mechanism.
   --
   --  Another format (referenced below as new format or gnatcheck
   --  qualification report) is more laconic and contains both compiler error
   --  messages and rule exemption warnings.

   -----------------------------------
   -- Old format of the report file --
   -----------------------------------

   All_Section_On : Boolean := True;
   --  This flag is set of by the first '-s{1|2|3} option together with setting
   --  OFF all the flags responsible for outputting a specific report file
   --  option

   Source_List_File_Name : constant String := "gnatcheck-source-list.out";
   --  Name of the file with all the argument file names created by
   --  gnatcheck if a used-provided file cannot be used for this.

   Rule_List_File_Name : constant String := "gnatcheck-rule-list.out";
   --  Name of the file with all the active rules with their actual parameters
   --  created by gnatcheck if a used-provided coding standard file cannot be
   --  used for this.

   ---------------------
   -- Diagnoses Table --
   ---------------------

   type Diag_Id is new Natural;
   No_Diag    : constant Diag_Id := Diag_Id'First;
   First_Diag : constant Diag_Id := No_Diag + 1;

   function No      (D : Diag_Id) return Boolean;
   function Present (D : Diag_Id) return Boolean;
   --  Check whether or not the argument is the Id of some existing diagnosis

   type Diagnosis_Record is record
      Rule          : Rule_Id;
      SF            : SF_Id;
      Diagnosis_Num : Diagnosis_Variant;

      Line          : Types.Physical_Line_Number;
      Col           : Types.Column_Number;
      --  We use GNAT types for line and column numbers

      Next_Diag           : Diag_Id;
      Prev_Diag           : Diag_Id;
      Next_Same_Rule_Diag : Diag_Id;
      Prev_Same_Rule_Diag : Diag_Id;
      --  For each source file, diagnostic messages are chained twice. First,
      --  we have the chain of all the rule violations detected for the current
      --  source, ordered by increasing the Sloc of the violation, and second,
      --  for each rule we have a chain of the violations of this particular
      --  rule, also ordered by increasing the Slocs

      Diag_Text : String_Loc;
      --  For the "rules" corresponding to the checks performed by the compiler
      --  (that is, general warnings, style warnings and restriction warnings)
      --  we do not have any diagnosis stored as a part of the corresponding
      --  rule, so we have to store the diagnostic messages as it is extracted
      --  from the compiler-generated information. This field is used to store
      --  such a diagnostic message

      Exempt_Justification : String_Loc;
      --  If set not to Nil_String_Loc, denotes the diagnoses for exempted rule
      --  and represents the justification of exemption.

      SLOC : String_Loc;
      --  We use the Line and Col fields to order diagnoses, and we use the
      --  SLOC field to put the full GNAT location in the report

   end record;

   package Rule_Violations is new Table.Table (
     Table_Component_Type => Diagnosis_Record,
     Table_Index_Type     => Diag_Id,
     Table_Low_Bound      => First_Diag,
     Table_Initial        => 10000,
     Table_Increment      => 100,
     Table_Name           => "diagnoses database");

   Diag_Table : Rule_Violations.Table_Ptr renames Rule_Violations.Table;

   ----------------------------------------
   --  File-Rule-Diagnosis Mapping Table --
   ----------------------------------------

   --  This table for each rule and for each source contains the links to the
   --  first and to the last detected diagnosis corresponding to the violation
   --  of the given rule in the given source. It also contains as its last row
   --  links to the first and to the latest detected violation of (any) rule
   --  detected in the given source.

   type Detected_Violations is record
      First : Diag_Id;
      Last  : Diag_Id;
   end record;
   --  An elementary entry in the Mapping Table. Contains the links to the
   --  first and to the last diagnosis for the given rule in the given file.

   No_Violation : constant Detected_Violations := (No_Diag, No_Diag);

   type File_Mapping_Type is array (Rule_Id range <>) of Detected_Violations;
   --  The diagnosis mapping for a single file

   type File_Mapping_Type_Access is access File_Mapping_Type;
   --  We do not know how many rules we will have, so we have to use dynamic
   --  mappings

   package Mapping_Table_Package is  new Table.Table (
     Table_Component_Type => File_Mapping_Type_Access,
     Table_Index_Type     => SF_Id,
     Table_Low_Bound      => First_SF_Id,
     Table_Initial        => 10000,
     Table_Increment      => 100,
     Table_Name           => "diagnoses mapping database");
   --  The Mapping Table. We have to use dynamic table here, because we do not
   --  know in advance how many needed sources may be added during the rule
   --  checking

   Mapping_Table : Mapping_Table_Package.Table_Ptr renames
     Mapping_Table_Package.Table;

   All_Diags : Rule_Id;
   --  The index of the last row in the table that contains the beginning and
   --  the end of all the diagnosis for the given file;

   Warning_Diags     : Rule_Id;
   Style_Diags       : Rule_Id;
   Restriction_Diags : Rule_Id;
   --  Rows to store compiler warnings, they immediately precede All_Diags

   function First_Diagnosis
     (SF            : SF_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id;

   function First_Diagnosis (SF : SF_Id) return Diag_Id;

   function First_Rule_Diagnosis
     (SF            : SF_Id;
      R             : Rule_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id;

   function First_Rule_Diagnosis
     (SF     : SF_Id;
      R      : Rule_Id)
      return   Diag_Id;

   function Last_Diagnosis
     (SF            : SF_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id;

   function Last_Diagnosis (SF : SF_Id) return Diag_Id;

   function Last_Rule_Diagnosis
     (SF            : SF_Id;
      R             : Rule_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id;
   pragma Unreferenced (Last_Rule_Diagnosis);

   function Last_Rule_Diagnosis
     (SF     : SF_Id;
      R      : Rule_Id)
      return   Diag_Id;

   function Next_Diagnosis
     (D             : Diag_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id;

   function Next_Same_Rule_Diagnosis
     (D             : Diag_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id;

   --  Returns the first (last, next) diagnostic stored for the given source
   --  (and for the given rule). If Exempted_Rule is ON, only diagnoses for
   --  exempted rules are considered, otherwise only diagnoses for non-exempted
   --  rules are considered. If a function does not have Exempted_Rule
   --  parameter, the exemption state of the rule is not taken into account.

   function Is_For_Exempted_Rule (D : Diag_Id) return Boolean;
   --  Tells if the argument diagnosis has been generated for exempted rule

   procedure Set_First_Diagnosis      (SF : SF_Id; D : Diag_Id);
   procedure Set_First_Rule_Diagnosis (SF : SF_Id; R : Rule_Id; D : Diag_Id);
   procedure Set_Last_Diagnosis       (SF : SF_Id; D : Diag_Id);
   procedure Set_Last_Rule_Diagnosis  (SF : SF_Id; R : Rule_Id; D : Diag_Id);
   --  Sets the first (last) diagnostic stored for the given source (and for
   --  the given rule)

   procedure Store_Rule_Violation_Internal
     (For_Rule      : Rule_Id;
      Line_Num      : Types.Physical_Line_Number;
      Col_Num       : Types.Column_Number;
      In_SF         : SF_Id;
      Justification : String_Loc;
      Diagnosis_Num : Diagnosis_Variant := 0;
      Diag_Text     : String_Loc        := Nil_String_Loc;
      Element_SLOC  : String_Loc        := Nil_String_Loc);
   --  This routine does the actual job for storing the diagnosis into
   --  diagnosis table. It needs to know only the rule, the file, the line and
   --  column to associate the diagnosis with and the diagnostic variant.
   --  This procedure also checks compiler messages for duplications and it
   --  does not store duplicated diagnoses.

   -----------------------
   -- Local subprograms --
   -----------------------

   --  The following subprograms are used to generate GNATCHECK report.
   --  (Should all of these routines be defined here?)

   procedure Generate_Regular_Report;
   procedure Generate_Qualification_Report;
   --  Umbrella procedures for generating different report formats.

   procedure Print_Report_Header;
   --  Generates the report header, including the date, tool version and
   --  tool command liner invocation sequence.

   procedure Print_Active_Rules;
   --  Generates the list of rules used (set as active) for the given
   --  gnatcheck run.

   procedure Print_Active_Rules_File;
   --  Prints the reference to the (actual argument or artificially created)
   --  file that contains the list of all the rules that are active for the
   --  given gnatcheck run

   procedure Print_File_List_File;
   --  Prints the reference to the (actual argument or artificially created)
   --  file that contains the list of all the files pacced to gnatcheck

   procedure Print_Disabled_Rules;
   --  Generate the list of rules currently defined in gnatcheck, but not used
   --  (set as disabled) for the given gnatcheck run.

   procedure Print_Source_List;
   --  Prints list of sources set as the arguments for the given gnatcheck run.

   procedure Print_Failure_Info;
   --  Prints info about non-fatal failures detected during gnatcheck run.

   procedure Print_Sections_Summary;
   --  Gives a short summary of the sections contained in the report.

   procedure Print_Exempted_Rules_Header;
   --  Generates the header of the section containing diagnoses for exempted
   --  rules.

   procedure Print_Section_1 (Exempted_Rules : Boolean := False);
   --  Print out the list of rule violations in compiler-like style (grouped
   --  by files with no specific order, and for each file diagnosis are
   --  ordered by the source locations of the wrong constructs). If
   --  Exempted_Rules is OFF, only diagnoses of non-exempted rules are printed,
   --  and the other way around. This procedure duplicates diagnoses into
   --  Stderr.

   procedure Print_Section_2 (Exempted_Rules : Boolean := False);
   --  Prints out the list of violations ordered by rules. For each rule
   --  violations detected in the same file are grouped together and are
   --  ordered  by the source locations of the wrong constructs.  If
   --  Exempted_Rules is OFF, only diagnoses of non-exempted rules are printed,
   --  and the other way around.

   procedure Print_Section_3 (Exempted_Rules : Boolean := False);
   --  Prints the list of violations where diagnoses are first ordered by
   --  rules, and for each file - by rules.  If Exempted_Rules is OFF, only
   --  diagnoses of non-exempted rules are printed, and the other way around.

   procedure Compute_Alignment
     (For_File       : SF_Id;
      Exempted_Rules : Boolean;
      Line_Pos       : out Natural;
      Col_Pos        : out Natural);
   --  For the diagnostic messages generated for  For_File computes the max
   --  number of positions for line and column number (the caller is
   --  responsible for the fact that diagnosis chain for For_File is not
   --  empty).  Exempted_Rules is ON, allignment is computed for doagnoses
   --  generataed for exempted rules, otherwise - for non-exempted rules.

   function Line_Col
     (D        : Diag_Id;
      Line_Pos : Natural := 0;
      Col_Pos  : Natural := 0)
      return     String;
   --  For the given diagnosis, returns the location in the form
   --  "line_number:coulumn_number". If Line_Pos and Col_Pos are not equal to
   --  zero, uses exactly Line_Pos positions for line_number and exactly
   --  Col_Pos positions for coulumn_number, raises Constraint_Error if the
   --  room is unsufficient

   function Line (D : Diag_Id) return Types.Physical_Line_Number;
   function Col (D : Diag_Id)  return Types.Column_Number;
   --  For the given diagnosis, return the line and column numbers

   function Text_Diag (D : Diag_Id) return String;
   --  Returns the text of the corresponding diagnosis message. This includes
   --  resolving the diagnosis variants, if needed

   function Strip_Column (SLOC : String) return String;
   --  Remove from the standard SLOC string the column number if
   --  No_Column_Num_In_Diagnoses is ON. returns the (unchanged) argument
   --  otherwise.

   -----------------------------
   --  New report file format --
   -----------------------------

   -----------------------
   -- Diagnoses storage --
   -----------------------

   type Diag_Message is record
      Text           : String_Access;
      Justification  : String_Access;
      Diagnosis_Kind : Diagnosis_Kinds;
      SF             : SF_Id;
      Num            : Positive;
      --  Is needed to order properly messages corresponding to the same SLOC.
   end record;

   function Get_Justification (Str : String) return String_Access;
   --  If Str is an empty string, returns null, otherwise returns
   --  new String'(Str).

   function Next_Message_Num return Positive;
   --  Returns next value for the Num field of diagnostic message

   function Diag_Is_Less_Then (L, R : Diag_Message) return Boolean;
   --  If L or R is null, raises Constraint_Error. Othervise lexicographically
   --  compares L.all and R.all provided that the prefix of these strings that
   --  denotes file name is converted to lower case.

   function Diag_Is_Equal (L, R : Diag_Message) return Boolean;
   --  If L or R is null, raises Constraint_Error. Othervise returns
   --  "="(L.all, R.all), where "=" is predefined equiality for String

   package All_Error_Messages_Storage is new Ada.Containers.Ordered_Sets
     (Element_Type => Diag_Message,
      "<"          => Diag_Is_Less_Then,
      "="          => Diag_Is_Equal);

   All_Error_Messages : All_Error_Messages_Storage.Set;

   Unused_Position : All_Error_Messages_Storage.Cursor;
   Unused_Inserted : Boolean;
   --  Needed as actuals for call to All_Error_Messages_Storage.Insert, are
   --  not used for anything

   procedure Count_Diagnoses (Position : All_Error_Messages_Storage.Cursor);
   --  Add 1 to the corresponding element of Error_Statistics global array
   --  depending on the kind of diagnosis stored under cursor.

   procedure Print_Out_Diagnoses;
   --  Duplicates diagnoses about non-exempted rule violations, exemption
   --  warnings and compiler error messages into Stderr. Up to Max_Diagnoses
   --  diagnoses are reported. If Max_Diagnoses equal to 0, all the diagnoses
   --  of these kinds are reported.

   -----------------------------------------
   -- Routines used to compose the report --
   -----------------------------------------

   procedure Print_Gnatcheck_Command_Line;
   --  Prints the gnatcheck command line. In case if gnatcheck has been
   --  called from the GNAT driver, prints the call to the GNAT driver, but not
   --  the gnatcheck call generated by the GNAT driver.

   procedure Print_Runtime;
   --  Prints the runtime version used for gnatcheck call. It is either the
   --  parameter of --RTS option used for (actual) gnatcheck call or the
   --  "<default>" string if --RTS parameter is not specified.

   procedure Print_Argument_Files_Summary;
   --  Prints the total numbers of: all the argument files, non-compilable
   --  files, files with no violations, files with violations, files with
   --  exempted violations only.

   function All_Sources_In_One_File return Boolean;
   --  Checks if all the argument sources are listed in a single user-provided
   --  file. In case if gnatcheck is called from the GNAT driver analyses
   --  the original call sequence for the GNAT driver.

   procedure Compute_Statisctics;
   --  Computes the number of violations and diagnoses of different kinds.
   --  Results are stoted in Error_Statistics global variable.

   procedure Print_Violation_Summary;
   --  Prints the total number of detected (non-exempted) violations and
   --  total number of exempted violations

   procedure Print_Violations (Exempted : Boolean);
   pragma Unreferenced (Print_Violations);
   --  Prints the information about detected violations. Depending on the
   --  actual parameter value, the information about exempted or non-exempted
   --  rules is printed out.

   procedure Print_Clean_Files;
   pragma Unreferenced (Print_Clean_Files);
   --  Prints out a list of files with no violation at all (neither
   --  non-exempted nor exempted).

   Diagnoses_To_Print : array (Exempted_Rule_Violation .. Compiler_Error)
     of Boolean := (others => False);

   procedure Print_Diagnoses;
   --  Prints into tool report file diagnoses stored in the global storage.
   --  Only messages of the kind(s) for that Diagnoses_To_Print is True are
   --  printed. Messages are alphabetically ordered, each message is printed
   --  only once

   procedure Print_Specified_Diagnoses
     (Position : All_Error_Messages_Storage.Cursor);
   --  If the diagnisis stored under cursor is of the rind for that
   --  Diagnoses_To_Print is true, prints this diagnosis into report file with
   --  indentation level 0 (???). Otherwise does nothing

   procedure Print_Non_Compilable_Files;
   pragma Unreferenced (Print_Non_Compilable_Files);
   --  Prints out a list of files that have not been successfully compiled.

   procedure Copy_User_Info;
   --  Copies into the report file the text from user-provided file.

   function Text_Justification (D : Diag_Id) return String;
   --  Returns the text of exemption justification

   -------------------------------
   -- Add_Line_To_Mapping_Table --
   -------------------------------

   procedure Add_Line_To_Mapping_Table is
   begin
      Mapping_Table_Package.Append
        (new File_Mapping_Type'(All_Rules.First .. All_Diags =>
                                No_Violation));
   end Add_Line_To_Mapping_Table;

   -----------------------------
   -- All_Sources_In_One_File --
   -----------------------------

   function All_Sources_In_One_File return Boolean is
      Result : Boolean;
   begin
      Result := not Individual_Files_Specified
              and then
                Arg_Source_File_Name /= "";

      if Result
        and then
         Getenv ("GNAT_DRIVER_COMMAND_LINE") /= null
      then
         --  Analyze GNAT driver call
         declare
            GNAT_Driver_Call : constant String :=
              Getenv ("GNAT_DRIVER_COMMAND_LINE").all;

            Word_Start : Natural := GNAT_Driver_Call'First;
            Word_End   : Natural := GNAT_Driver_Call'First;

            First_Idx :           Natural := GNAT_Driver_Call'First;
            Last_Idx  : constant Natural := GNAT_Driver_Call'Last;

            Num_Of_Arg_Files : Natural := 0;

            procedure Set_Word;
            --  Select Word_Start and Word_End to point to the first word in
            --  GNAT_Driver_Call (First_Idx .. Last_Idx), and then set
            --  First_Idx to Word_End + 1. If there is no word any more in
            --  GNAT_Driver_Call (First_Idx .. Last_Idx), set Word_Start to
            --  Last_Idx + 1

            procedure Set_Word is
            begin
               Word_Start := Last_Idx + 1;

               for J in First_Idx .. Last_Idx loop
                  if GNAT_Driver_Call (J) /= ' ' then
                     Word_Start := J;
                     exit;
                  end if;
               end loop;

               if Word_Start <= Last_Idx then
                  Word_End := Last_Idx;

                  for J in Word_Start .. Last_Idx loop
                     if GNAT_Driver_Call (J) = ' ' then
                        Word_Start := J - 1;
                        exit;
                     end if;
                  end loop;

                  First_Idx := Word_End + 1;
               end if;

            end Set_Word;

         begin
            --  Just in case:
            if GNAT_Driver_Call = "" then
               goto Result_Detected;
            end if;

            --  First, case when -U option is given:
            if Index (GNAT_Driver_Call, "-U") /= 0 then
               Result := False;
               goto Result_Detected;
            end if;

            --  Here we have to parse the string.
            --  Skip the call sequence up to the space after 'check' - there is
            --  nothing interesting in it

            First_Idx :=
              Index (Ada.Characters.Handling.To_Lower (GNAT_Driver_Call),
                    "check");
            First_Idx := First_Idx + 5;

            Set_Word;

            while Word_Start <= Last_Idx loop
               if GNAT_Driver_Call (Word_Start) = '-' then
                  case GNAT_Driver_Call (Word_Start + 1) is
                     when 'a' |
                          'd' |
                          'h' |
                          'm' |
                          'q' |
                          't' |
                          'v' |
                          's' |
                          'l' |
                          '-' =>
                        --  Just skip the option, either gnatcheck's or
                        --  GNAT driver's
                        null;
                     when 'c' | -- 'cargs'
                          'r' => -- 'rules'
                        --  No information about argument sources in the rest
                        --  of the call sequence
                        exit;
                     when 'P' |
                          'o' =>
                        --  Project file: we can have either '-P prj' or -Pprj
                        --  Specification of the output file: we may have
                        --  either '-o bla' or -o=foo
                        if Word_Start + 1 = Word_End then
                           Set_Word;
                        else
                           null;
                        end if;

                     when 'f' =>
                        --  Specification of the argument file list: we may
                        --  have either -files=foo or '-files bar'

                        Num_Of_Arg_Files := Num_Of_Arg_Files + 1;

                        if Num_Of_Arg_Files > 1 then
                           Result := False;
                           goto Result_Detected;
                        end if;

                        if Word_Start + 5 = Word_End then
                           Set_Word;
                        end if;

                     when others =>
                        Error ("bug in comman line analysis");
                        Error ("report a problem to report@adacore.com");
                        raise Fatal_Error;
                  end case;
               else
                  --  Definitely an explicitely specified file name, so
                  Result := False;
                  goto Result_Detected;
               end if;

               Set_Word;
            end loop;

            if Result and then Num_Of_Arg_Files = 0 then
               --  Call to the GNAT driver with no argument files specified,
               --  all the project files should be processed
               Result := False;
            end if;
         end;
      end if;

      <<Result_Detected>> return Result;
   end All_Sources_In_One_File;

   ---------
   -- Col --
   ---------

   function Col (D : Diag_Id)  return Types.Column_Number is
   begin
      return Diag_Table (D).Col;
   end Col;

   ------------------------
   --  Compute_Alignment --
   ------------------------

   procedure Compute_Alignment
     (For_File       : SF_Id;
      Exempted_Rules : Boolean;
      Line_Pos       : out Natural;
      Col_Pos        : out Natural)
   is
      Max_Pos     : Natural;
      Max_Col_Num : Types.Column_Number := 1;
      Next_Diag   : Diag_Id;
   begin
      --  Max line positions
      Max_Pos  := Positive (Line (Last_Diagnosis (For_File, Exempted_Rules)));
      Line_Pos := 0;

      while Max_Pos > 0 loop
         Line_Pos := Line_Pos + 1;
         Max_Pos  := Max_Pos / 10;
      end loop;

      --  Max column positions

      Next_Diag := First_Diagnosis (For_File, Exempted_Rules);

      while Present (Next_Diag) loop

         if Types.">" (Col (Next_Diag), Max_Col_Num) then
            Max_Col_Num :=  Col (Next_Diag);
         end if;

         Next_Diag := Next_Diagnosis (Next_Diag, Exempted_Rules);
      end loop;

      Max_Pos := Positive (Max_Col_Num);
      Col_Pos := 0;

      while Max_Pos > 0 loop
         Col_Pos := Col_Pos + 1;
         Max_Pos := Max_Pos / 10;
      end loop;

   end Compute_Alignment;

   -------------------------
   -- Compute_Statisctics --
   -------------------------

   procedure Compute_Statisctics is
   begin
      All_Error_Messages_Storage.Iterate
        (Container => All_Error_Messages,
         Process   => Count_Diagnoses'Access);
   end Compute_Statisctics;

   --------------------
   -- Copy_User_Info --
   --------------------

   procedure Copy_User_Info is
      Max_Line_Len : constant Positive := 1024;
      Line_Buf     :          String (1 .. Max_Line_Len);
      Line_Len     :          Natural;
      User_File    :          Ada.Text_IO.File_Type;
   begin
      --  Very simple-minded implementation...

      Open (File => User_File,
            Mode => In_File,
            Name => User_Info_File_Full_Path.all);

      Get_Line (File => User_File,
                Item => Line_Buf,
                Last => Line_Len);

      while not End_Of_File (User_File) loop
         Report (Line_Buf (1 .. Line_Len));

         Get_Line (File => User_File,
                   Item => Line_Buf,
                   Last => Line_Len);
      end loop;

      Close (User_File);
   exception
      when E : others =>

         Report_EOL;
         Report ("cannot sucsessfully copy information " &
                 "from " & User_Info_File.all);

         if Is_Open (User_File) then
            Close (User_File);
         end if;

         Error ("cannot copy information from " & User_Info_File.all &
                " into report file");

         Error_No_Tool_Name (Ada.Exceptions.Exception_Information (E));
   end Copy_User_Info;

   ---------------------
   -- Count_Diagnoses --
   ---------------------

   procedure Count_Diagnoses (Position : All_Error_Messages_Storage.Cursor) is
   begin
      Error_Statistics
       (All_Error_Messages_Storage.Element (Position).Diagnosis_Kind) :=
      Error_Statistics
       (All_Error_Messages_Storage.Element (Position).Diagnosis_Kind) + 1;

   end Count_Diagnoses;

   --------------------------
   -- Create_Mapping_Table --
   --------------------------

   procedure Create_Mapping_Table is
      Rule_Number : constant Rule_Id := All_Rules.Last + 1 + 3;
      --  "+1" - for the total number of violations
      --  "+3" - for general warnings, style warnings and restriction warnings
   begin
      Mapping_Table_Package.Set_Last (Last_Argument_Source);

      for J in First_SF_Id .. Last_Argument_Source loop
         Mapping_Table (J) :=
           new File_Mapping_Type'(All_Rules.First .. Rule_Number =>
               No_Violation);
      end loop;

      All_Diags := Rule_Number;

      Warning_Diags     := All_Diags - 3;
      Style_Diags       := All_Diags - 2;
      Restriction_Diags := All_Diags - 1;

   end Create_Mapping_Table;

   -------------------
   -- Diag_Is_Equal --
   -------------------

   function Diag_Is_Equal (L, R : Diag_Message) return Boolean is
   begin
      return L.Text.all = R.Text.all;
   end Diag_Is_Equal;

   -----------------------
   -- Diag_Is_Less_Then --
   -----------------------

   function Diag_Is_Less_Then (L, R : Diag_Message) return Boolean is
      L_Start : constant Natural := L.Text'First;
      R_Start : constant Natural := R.Text'First;

      L_SLOC_End : constant Natural := Index (L.Text.all, ": ");
      R_SLOC_End : constant Natural := Index (R.Text.all, ": ");

   begin
      return SLOC_Less_Then (L.Text (L_Start .. L_SLOC_End - 1),
                             R.Text (R_Start .. R_SLOC_End - 1))
            or else
             (not Diag_Is_Equal (L, R)
             and then
              L.Text (L_Start .. L_SLOC_End - 1) =
              L.Text (L_Start .. L_SLOC_End - 1)
             and then
              L.Num < R.Num);

   end Diag_Is_Less_Then;

   --------------------------------------------------
   -- Diag_Srorage_Debug_Image (new report format) --
   --------------------------------------------------

   procedure Diag_Srorage_Debug_Image is
      procedure Debug_Image (Position : All_Error_Messages_Storage.Cursor);

      procedure Debug_Image (Position : All_Error_Messages_Storage.Cursor) is
         Tmp : constant Diag_Message :=
           All_Error_Messages_Storage.Element (Position);
      begin
         Info ("Text");
         Info (Tmp.Text.all);

         Info ("Justification");

         if Tmp.Justification = null then
            Info ("<no justification>");
         else
            Info (Tmp.Justification.all);
         end if;

         Info ("Diagnosis_Kind: " & Tmp.Diagnosis_Kind'Img);

         Info ("SF            :" & Tmp.SF'Img);
         Info ("Num           :" & Tmp.Num'Img);
         Info ("");
      end Debug_Image;

   begin
      Info ("***Diagnoses storage debug image start");

      All_Error_Messages_Storage.Iterate
        (Container => All_Error_Messages,
         Process   => Debug_Image'Access);

      Info ("***Diagnoses storage debug image end");
   end Diag_Srorage_Debug_Image;

   ----------------------------------------------------
   -- Diag_Structure_Debug_Image (old report format) --
   ----------------------------------------------------

   procedure Diag_Structure_Debug_Image is
      Ident_String : constant String := "   ";
      procedure Print_Diag_Node (N : Diag_Id);
      --  Prints out one diag node

      procedure Pring_File_Mapping (SF : SF_Id);
      --  Prints out mapping for the argument file

      procedure Print_Diag_Node (N : Diag_Id) is
         Rule : constant Rule_Id := Diag_Table (N).Rule;
      begin
         Put_Line ("Diag_Id =" & N'Img);
         Put (Ident_String);
         Put ("Rule = ");

         if Rule = Warning_Diags then
            Put_Line ("Compiler warnings");
         elsif Rule = Style_Diags then
            Put_Line ("Compiler style checks");
         elsif Rule = Restriction_Diags then
            Put_Line ("Compiler restrictions");
         else
            Put_Line (Rule_Name (All_Rules.Table (Rule).all));
         end if;

         Put (Ident_String);
         Put_Line ("SF =  " & Diag_Table (N).SF'Img);

         Put (Ident_String);
         Put_Line ("Line =" & Diag_Table (N).Line'Img &
                  " Col=" & Diag_Table (N).Col'Img);

         Put (Ident_String);
         Put_Line ("Next_Diag =           " & Diag_Table (N).Next_Diag'Img);

         Put (Ident_String);
         Put_Line ("Prev_Diag =           " & Diag_Table (N).Prev_Diag'Img);

         Put (Ident_String);
         Put_Line ("Next_Same_Rule_Diag = " &
                   Diag_Table (N).Next_Same_Rule_Diag'Img);

         Put (Ident_String);
         Put_Line ("Prev_Same_Rule_Diag = " &
                   Diag_Table (N).Prev_Same_Rule_Diag'Img);

         if Is_For_Exempted_Rule (N) then
            Put (Ident_String);
            Put_Line ("Is exempted");
         end if;

      end Print_Diag_Node;

      procedure Pring_File_Mapping (SF : SF_Id) is
         procedure Print_Rule_Mapping (R : Rule_Id; SF : SF_Id);
         --  Prints out the concrete diagnosis mapping

         procedure Print_Rule_Mapping (R : Rule_Id; SF : SF_Id) is
         begin

            Put (Ident_String);

            if R <= All_Rules.Last then
               Put_Line ("Rule = " &
                         Rule_Name (All_Rules.Table (R).all));
            else
               Put_Line ("All Rules");
            end if;

            Put (Ident_String);
            Put (Ident_String);
            Put_Line ("First =" & Mapping_Table (SF) (R) .First'Img &
                      " Last =" & Mapping_Table (SF) (R).Last'Img);

         end Print_Rule_Mapping;
      begin
         Put_Line ("Mapping for file " & SF'Img);

         for J in Mapping_Table (SF)'Range loop
            Print_Rule_Mapping (J, SF);
         end loop;

      end Pring_File_Mapping;

   begin
      Put_Line ("**** Diag_Table start *****");

      for J in First_Diag .. Rule_Violations.Last loop
         Print_Diag_Node (J);
      end loop;

      Put_Line ("**** Diag_Table end *****");

      Put_Line ("**** Mappint_Table start *****");

      for J in First_SF_Id .. Mapping_Table_Package.Last loop
         Pring_File_Mapping (J);
      end loop;

      Put_Line ("**** Mapping_Table end *****");

   end Diag_Structure_Debug_Image;

   ---------------------
   -- First_Diagnosis --
   ---------------------

   function First_Diagnosis
     (SF            : SF_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id
   is
      Result : Diag_Id := Mapping_Table (SF) (All_Diags).First;
   begin

      if Exemption_On then

         while Present (Result)
           and then
               Is_For_Exempted_Rule (Result) /= Exempted_Rule
         loop
            Result := Diag_Table (Result).Next_Diag;
         end loop;

      end if;

      return Result;
   end First_Diagnosis;

   function First_Diagnosis (SF : SF_Id) return Diag_Id is
   begin
      return Mapping_Table (SF) (All_Diags).First;
   end First_Diagnosis;

   --------------------------
   -- First_Rule_Diagnosis --
   --------------------------

   function First_Rule_Diagnosis
     (SF            : SF_Id;
      R             : Rule_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id
   is
      Result : Diag_Id := Mapping_Table (SF) (R).First;
   begin

      if Exemption_On then

         while Present (Result)
           and then
               Is_For_Exempted_Rule (Result) /= Exempted_Rule
         loop
            Result := Diag_Table (Result).Next_Same_Rule_Diag;
         end loop;

      end if;

      return Result;
   end First_Rule_Diagnosis;

   function First_Rule_Diagnosis
     (SF     : SF_Id;
      R      : Rule_Id)
      return   Diag_Id
   is
   begin
      return  Mapping_Table (SF) (R).First;
   end First_Rule_Diagnosis;

   -----------------------------------
   -- Generate_Qualification_Report --
   -----------------------------------

   procedure Generate_Qualification_Report is
   begin
      Compute_Statisctics;

      --  OVERVIEW
      if not Short_Report then

         Print_Report_Header;
         Print_Active_Rules_File;
         Print_File_List_File;
         Print_Argument_Files_Summary;
         Report_EOL;
         Print_Violation_Summary;
         Print_Failure_Info;

         --  2. DETECTED EXEMPTED RULE VIOLATIONS
         Report_EOL;
         Report ("2. Exempted Coding Standard Violation");
         Report_EOL;
      end if;

      if Error_Statistics (Exempted_Rule_Violation) > 0 then
         Diagnoses_To_Print :=
           (Exempted_Rule_Violation               => True,
            Exemption_Warning                     => False,
            Non_Exempted_Rule_Violation           => False,
            Non_Exempted_Compiler_Check_Violation => False,
            Compiler_Error                        => False);

         Print_Diagnoses;
      else
         Report ("no exempted violations detected", 1);
      end if;

      if not Short_Report then
         Report_EOL;
      end if;

      if not Short_Report then
         Report ("3. Non-exempted Coding Standard Violations");
         Report_EOL;
      end if;

      if Error_Statistics (Non_Exempted_Rule_Violation) > 0
        or else
         Error_Statistics (Non_Exempted_Compiler_Check_Violation) > 0
      then
         Diagnoses_To_Print :=
           (Exempted_Rule_Violation               => False,
            Exemption_Warning                     => False,
            Non_Exempted_Rule_Violation           => True,
            Non_Exempted_Compiler_Check_Violation => True,
            Compiler_Error                        => False);

         Print_Diagnoses;
      else
         Report ("no non-exempted violations detected", 1);
      end if;

      if not Short_Report then
         Report_EOL;
      end if;

      if not Short_Report then
         Report ("4. Rule exemption problems");
         Report_EOL;
      end if;

      if Error_Statistics (Exemption_Warning) > 0 then
         Diagnoses_To_Print :=
           (Exempted_Rule_Violation               => False,
            Exemption_Warning                     => True,
            Non_Exempted_Rule_Violation           => False,
            Non_Exempted_Compiler_Check_Violation => False,
            Compiler_Error                        => False);

         Print_Diagnoses;
      else
         Report ("no rule exemption problem detected", 1);
      end if;

      if not Short_Report then
         Report_EOL;
      end if;

      if not Short_Report then
         Report ("5. Language violations");
         Report_EOL;
      end if;

      if Error_Statistics (Compiler_Error) > 0 then
         Diagnoses_To_Print :=
           (Exempted_Rule_Violation               => False,
            Exemption_Warning                     => False,
            Non_Exempted_Rule_Violation           => False,
            Non_Exempted_Compiler_Check_Violation => False,
            Compiler_Error                        => True);

         Print_Diagnoses;
      else
         Report ("no language violation detected", 1);
      end if;

         --  User-defined part

      if not Short_Report then

         Report_EOL;

         if User_Info_File /= null then
            Report ("6. Additional Information");
            Report_EOL;
            Copy_User_Info;
            Report_EOL;

         end if;

      end if;

      --  Sending the diagnoses into Stderr.
      if not Quiet_Mode then
         Print_Out_Diagnoses;
      end if;

--      Diag_Srorage_Debug_Image;
   end Generate_Qualification_Report;

   -----------------------------
   -- Generate_Regular_Report --
   -----------------------------

   procedure Generate_Regular_Report is
   begin

      if not Short_Report then
         Print_Report_Header;
         Print_Active_Rules;
         Print_Disabled_Rules;
         Print_Source_List;
         Print_Failure_Info;
         Print_Sections_Summary;
      end if;

      if Output_Section_1
        or else
         not Quiet_Mode
      then
         Print_Section_1;
      end if;

      if Output_Section_2 then
         Print_Section_2;
      end if;

      if Output_Section_3 then
         Print_Section_3;
      end if;

      if Exemption_On
        and then
         Print_Exemption_Section
      then
         Print_Exempted_Rules_Header;

         if Output_Section_1 then
            Print_Section_1 (Exempted_Rules => True);
         end if;

         if Output_Section_2 then
            Print_Section_2 (Exempted_Rules => True);
         end if;

         if Output_Section_3 then
            Print_Section_3 (Exempted_Rules => True);
         end if;
      end if;
   end Generate_Regular_Report;

   ---------------------
   -- Generate_Report --
   ---------------------

   procedure Generate_Report is
   begin
      if Qualification_Report then
         Generate_Qualification_Report;
      else
         Generate_Regular_Report;
      end if;
   end Generate_Report;

   -----------------------
   -- Get_Justification --
   -----------------------

   function Get_Justification (Str : String) return String_Access is
   begin
      if Str = "" then
         return null;
      else
         return new String'(Str);
      end if;
   end Get_Justification;

   --------------------------
   -- Is_For_Exempted_Rule --
   --------------------------

   function Is_For_Exempted_Rule (D : Diag_Id) return Boolean is
   begin
      return Diag_Table (D).Exempt_Justification /= Nil_String_Loc;
   end Is_For_Exempted_Rule;

   --------------------
   -- Last_Diagnosis --
   --------------------

   function Last_Diagnosis
     (SF            : SF_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id
   is
      Result : Diag_Id := Mapping_Table (SF) (All_Diags).Last;
   begin

      if Exemption_On then

         while Present (Result)
           and then
               Is_For_Exempted_Rule (Result) /= Exempted_Rule
         loop
            Result := Diag_Table (Result).Prev_Diag;
         end loop;

      end if;

      return Result;
   end Last_Diagnosis;

   function Last_Diagnosis (SF : SF_Id) return Diag_Id is
   begin
      return Mapping_Table (SF) (All_Diags).Last;
   end Last_Diagnosis;

   -------------------------
   -- Last_Rule_Diagnosis --
   -------------------------

   function Last_Rule_Diagnosis
     (SF            : SF_Id;
      R             : Rule_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id
   is
      Result : Diag_Id := Mapping_Table (SF) (R).Last;
   begin

      if Exemption_On then

         while Present (Result)
           and then
               Is_For_Exempted_Rule (Result) /= Exempted_Rule
         loop
            Result := Diag_Table (Result).Prev_Same_Rule_Diag;
         end loop;

      end if;

      return Result;
   end Last_Rule_Diagnosis;

   function Last_Rule_Diagnosis
     (SF     : SF_Id;
      R      : Rule_Id)
      return   Diag_Id
   is
   begin
      return Mapping_Table (SF) (R).Last;
   end Last_Rule_Diagnosis;

   --------------
   -- Line_Col --
   --------------

   function Line_Col
     (D        : Diag_Id;
      Line_Pos : Natural := 0;
      Col_Pos  : Natural := 0)
      return     String
   is
      Line_Str : constant String := Trim (Diag_Table (D).Line'Img, Left);
      Col_Str  : constant String := Trim (Diag_Table (D).Col'Img, Left);
   begin
      return
        (1 .. Line_Pos - Line_Str'Length => ' ') &
        Line_Str                                 &
        ':'                                      &
        (1 .. Col_Pos - Col_Str'Length => ' ')   &
        Col_Str;
   end Line_Col;

   ----------
   -- Line --
   ----------

   function Line (D : Diag_Id) return Types.Physical_Line_Number is
   begin
      return Diag_Table (D).Line;
   end Line;

   --------------------
   -- Next_Diagnosis --
   --------------------

   function Next_Diagnosis
     (D             : Diag_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id
   is
      Result : Diag_Id := Diag_Table (D).Next_Diag;
   begin

      if Exemption_On then

         while Present (Result)
           and then
               Is_For_Exempted_Rule (Result) /= Exempted_Rule
         loop
            Result := Diag_Table (Result).Next_Diag;
         end loop;

      end if;

      return Result;
   end Next_Diagnosis;

   ----------------------
   -- Next_Message_Num --
   ----------------------

   Next_Message_Num_Value : Natural := 0;

   function Next_Message_Num return Positive is
   begin
      Next_Message_Num_Value := Next_Message_Num_Value + 1;
      return Next_Message_Num_Value;
   end Next_Message_Num;

   ------------------------------
   -- Next_Same_Rule_Diagnosis --
   ------------------------------

   function Next_Same_Rule_Diagnosis
     (D             : Diag_Id;
      Exempted_Rule : Boolean)
      return          Diag_Id
   is
      Result : Diag_Id := Diag_Table (D).Next_Same_Rule_Diag;
   begin

      if Exemption_On then

         while Present (Result)
           and then
               Is_For_Exempted_Rule (Result) /= Exempted_Rule
         loop
            Result := Diag_Table (Result).Next_Same_Rule_Diag;
         end loop;

      end if;

      return Result;
   end Next_Same_Rule_Diagnosis;

   --------
   -- No --
   --------

   function No (D : Diag_Id) return Boolean is
   begin
      return D not in First_Diag .. Rule_Violations.Last;
   end No;

   -------------
   -- Present --
   -------------

   function Present (D : Diag_Id) return Boolean is
   begin
      return D in First_Diag .. Rule_Violations.Last;
   end Present;

   ------------------------
   -- Print_Active_Rules --
   ------------------------

   procedure Print_Active_Rules is
   begin
      Report ("coding standard (applied rules):");

      for Rule in All_Rules.First .. All_Rules.Last loop

         if All_Rules.Table (Rule).Diagnosis /= null
          and then
            Is_Enable (All_Rules.Table (Rule).all)
         then
            --  Note, that if a rule does not have its own diagnoses, this
            --  means that it is implemented by some other rules, so it
            --  should not go into the report

            Print_Rule (All_Rules.Table (Rule).all, 1);
            Report_EOL;
         end if;
      end loop;

      Report_EOL;

      --  Compiler-made checks:

      if Use_gnaty_Option then
         Report_No_EOL ("Compiler style checks: ", 1);
         Report (Get_Style_Option);
         Report_EOL;
      end if;

      if Use_gnatw_Option then
         Report_No_EOL ("Compiler warnings: ", 1);
         Report (Get_Specified_Warning_Option);
         Report_EOL;
      end if;

      if Check_Restrictions then
         Report ("Compiler-checked restrictions:", 1);
         Print_Active_Restrictions (2);
         Report_EOL;
      end if;

   end Print_Active_Rules;

   -----------------------------
   -- Print_Active_Rules_File --
   -----------------------------

   procedure Print_Active_Rules_File is
      Rule_List_File : Ada.Text_IO.File_Type;
   begin
      Report_No_EOL ("coding standard   : ");

      if not Individual_Rules_Set
        and then
         Rule_File_Name /= null
      then
         Report (Rule_File_Name.all);
      else
         --  Creating the list of active rules in Rule_List_File_Name

         if Is_Regular_File
           (".." & Directory_Separator & Rule_List_File_Name)
         then
            Open
              (Rule_List_File,
               Out_File,
               ".." & Directory_Separator & Rule_List_File_Name);
         else
            Create
              (Rule_List_File,
               Out_File,
               ".." & Directory_Separator & Rule_List_File_Name);
         end if;

         for Rule in All_Rules.First .. All_Rules.Last loop

            if All_Rules.Table (Rule).Diagnosis /= null
             and then
               Is_Enable (All_Rules.Table (Rule).all)
            then
               --  Note, that if a rule does not have its own diagnoses, this
               --  means that it is implemented by some other rules, so it
               --  should not go into the report

               Print_Rule_To_File (All_Rules.Table (Rule).all, Rule_List_File);
               New_Line (Rule_List_File);
            end if;
         end loop;

         New_Line (Rule_List_File);

         --  Compiler-made checks:

         if Use_gnaty_Option then
            New_Line (Rule_List_File);
            Put_Line (Rule_List_File, "-- Compiler style checks:");
            Put      (Rule_List_File, "+RStyle_Checks : ");
            Put_Line (Rule_List_File,
                      Get_Style_Option
                        (Get_Style_Option'First + 6 .. Get_Style_Option'Last));
         end if;

         if Use_gnatw_Option then
            New_Line (Rule_List_File);
            Put_Line (Rule_List_File, "--  Compiler warnings:");
            Put      (Rule_List_File, "+RWarnings : ");
            Put_Line (Rule_List_File,
                      Get_Specified_Warning_Option
                        (Get_Specified_Warning_Option'First + 6 ..
                         Get_Specified_Warning_Option'Last));
         end if;

         if Check_Restrictions then
            New_Line (Rule_List_File);
            Put_Line (Rule_List_File, "--  Compiler restrictions:");
            Print_Active_Restrictions_To_File (Rule_List_File);
         end if;

         Close (Rule_List_File);

         Report (Rule_List_File_Name);
      end if;

   end Print_Active_Rules_File;

   ----------------------------------
   -- Print_Argument_Files_Summary --
   ----------------------------------

   procedure Print_Argument_Files_Summary is
      Checked_Sources                  : Natural := 0;
      Non_Compilable_Sources           : Natural := 0;
      Fully_Compliant_Sources          : Natural := 0;
      Sources_With_Violations          : Natural := 0;
      Sources_With_Exempted_Violations : Natural := 0;
   begin

      for SF in First_SF_Id .. Last_Argument_Source loop
         if Source_Status (SF) = Not_A_Legal_Source then
            Non_Compilable_Sources := Non_Compilable_Sources + 1;
         else
            Checked_Sources := Checked_Sources + 1;
         end if;

         if Present (First_Diagnosis (SF, Exempted_Rule => False)) then
            Sources_With_Violations := Sources_With_Violations + 1;
         elsif Present (First_Diagnosis (SF, Exempted_Rule => True)) then
            Sources_With_Exempted_Violations :=
              Sources_With_Exempted_Violations + 1;
         elsif Source_Status (SF) = Processed then
            Fully_Compliant_Sources := Fully_Compliant_Sources + 1;
         end if;

      end loop;

      Report ("1. Summary");
      Report_EOL;

      Report ("fully compliant sources               :" &
              Fully_Compliant_Sources'Img, 1);
      Report ("sources with exempted violations only :" &
              Sources_With_Exempted_Violations'Img, 1);
      Report ("sources with non-exempted violations  :" &
              Sources_With_Violations'Img, 1);
      Report ("non compilable sources                :" &
              Non_Compilable_Sources'Img, 1);
      Report ("total sources                         :" &
              Last_Argument_Source'Img, 1);

      pragma Assert (Checked_Sources =
                       Fully_Compliant_Sources +
                       Sources_With_Violations +
                       Sources_With_Exempted_Violations);
      pragma Assert (Natural (Last_Argument_Source) =
                       Checked_Sources + Non_Compilable_Sources);
   end Print_Argument_Files_Summary;

   -----------------------
   -- Print_Clean_Files --
   -----------------------

   procedure Print_Clean_Files is
      Next_Diag         : Diag_Id;
      Nothing_To_Report : Boolean := True;
   begin

      for SF in First_SF_Id .. Last_Source loop
         if Source_Status (SF) = Processed then
            Next_Diag := First_Diagnosis (SF, True);

            if No (Next_Diag) then
               Next_Diag := First_Diagnosis (SF, False);

               if No (Next_Diag) then
                  Report (Short_Source_Name (SF), 1);
                  Nothing_To_Report := False;
               end if;
            end if;
         end if;
      end loop;

      if Nothing_To_Report then
         Report ("No files without violations", 1);
      end if;
   end Print_Clean_Files;

   ---------------------
   -- Print_Diagnoses --
   ---------------------

   procedure Print_Diagnoses is
   begin
      All_Error_Messages_Storage.Iterate
        (Container => All_Error_Messages,
         Process   => Print_Specified_Diagnoses'Access);
   end Print_Diagnoses;

   --------------------------
   -- Print_Disabled_Rules --
   --------------------------

   procedure Print_Disabled_Rules is
   begin
      Report ("Disabled rules:");

      for Rule in All_Rules.First .. All_Rules.Last loop

         if All_Rules.Table (Rule).Diagnosis /= null
          and then
            not Is_Enable (All_Rules.Table (Rule).all)
         then
            --  Note, that if a rule does not have its own diagnoses, this
            --  means that it is implemented by some other rules, so it
            --  should not go into the report
            Report_No_EOL
              ("(" & Rule_Name (All_Rules.Table (Rule).all) & ") ", 1);
            Report (All_Rules.Table (Rule).Help_Info.all);
         end if;
      end loop;

      Report_EOL;

      --  Compiler-made checks:

      if not Use_gnaty_Option then
         Report ("No active compiler style check", 1);
      end if;

      if not Use_gnatw_Option then
         Report ("No active compiler warning:", 1);
      end if;

      if not Check_Restrictions then
         Report ("No active compiler-checked restriction", 1);
      end if;

      Report_EOL;

   end Print_Disabled_Rules;

   ---------------------------------
   -- Print_Exempted_Rules_Header --
   ---------------------------------

   procedure Print_Exempted_Rules_Header is
   begin
      Report_EOL;
      Report ("============== Start Exempted Rules Section  ===============");
      Report ("   This section contains diagnoses for exempted rules ");
      Report_EOL;

   end Print_Exempted_Rules_Header;

   ------------------------
   -- Print_Failure_Info --
   ------------------------

   procedure Print_Failure_Info is
   begin

      if Tool_Failures > 0 then
         Report ("Total gnatcheck failures:" & Tool_Failures'Img);
         Report_EOL;
      end if;

   end Print_Failure_Info;

   --------------------------
   -- Print_File_List_File --
   --------------------------

   procedure Print_File_List_File is
      Source_List_File : Ada.Text_IO.File_Type;
   begin
      Report_No_EOL ("list of sources   : ");

      if All_Sources_In_One_File then
         Report (Arg_Source_File_Name);
      else
         --  Creating the list of processed sources in Source_List_File_Name

         if Is_Regular_File
           (".." & Directory_Separator & Source_List_File_Name)
         then
            Open
              (Source_List_File,
               Out_File,
               ".." & Directory_Separator & Source_List_File_Name);
         else
            Create
              (Source_List_File,
               Out_File,
               ".." & Directory_Separator & Source_List_File_Name);
         end if;

         for SF in First_SF_Id .. Last_Argument_Source loop
            Put_Line (Source_List_File, Short_Source_Name (SF));
         end loop;

         Close (Source_List_File);

         Report (Source_List_File_Name);
      end if;

      Report_EOL;
   end Print_File_List_File;

   ----------------------------------
   -- Print_Gnatcheck_Command_Line --
   ----------------------------------

   procedure Print_Gnatcheck_Command_Line is
      GNAT_Driver_Call : constant String_Access :=
       Getenv ("GNAT_DRIVER_COMMAND_LINE");

   begin
      if GNAT_Driver_Call /= null
       and then
         GNAT_Driver_Call.all /= ""
      then
         Report (GNAT_Driver_Call.all);
      else
         Report_No_EOL (Ada.Command_Line.Command_Name);

         for Arg in 1 .. Ada.Command_Line.Argument_Count loop
            Report_No_EOL (" " & Ada.Command_Line.Argument (Arg));
         end loop;

         Report_EOL;
      end if;

   end Print_Gnatcheck_Command_Line;

   --------------------------------
   -- Print_Non_Compilable_Files --
   --------------------------------

   procedure Print_Non_Compilable_Files is
   begin
      for SF in First_SF_Id .. Last_Argument_Source loop
         if Source_Status (SF) = Not_A_Legal_Source then
            Report (Short_Source_Name (SF), 1);
         end if;
      end loop;
   end Print_Non_Compilable_Files;

   -------------------------
   -- Print_Out_Diagnoses --
   -------------------------

   procedure Print_Out_Diagnoses is
      Diagnoses_Reported : Natural := 0;
      Limit_Exceeded : Boolean := False;

      procedure Counted_Print_Diagnosis
        (Position : All_Error_Messages_Storage.Cursor);

      procedure Counted_Print_Diagnosis
        (Position : All_Error_Messages_Storage.Cursor)
      is
      begin
         if not Limit_Exceeded then
            if Diagnoses_Reported > Max_Diagnoses then
               Limit_Exceeded := True;
               Info ("Maximum diagnoses reached, "&
                     "see the report file for full details");
            else
               if Diagnoses_To_Print
                    (All_Error_Messages_Storage.Element (Position).
                       Diagnosis_Kind)
               then
                  Diagnoses_Reported := Diagnoses_Reported + 1;
                  Info (All_Error_Messages_Storage.Element (Position).
                    Text.all);
               end if;
            end if;
         end if;
      end Counted_Print_Diagnosis;

   begin
      Diagnoses_To_Print :=
        (Exempted_Rule_Violation               => False,
         Exemption_Warning                     => False,
         Non_Exempted_Rule_Violation           => True,
         Non_Exempted_Compiler_Check_Violation => True,
         Compiler_Error                        => True);

      All_Error_Messages_Storage.Iterate
        (Container => All_Error_Messages,
         Process   => Counted_Print_Diagnosis'Access);
   end Print_Out_Diagnoses;

   -------------------------
   -- Print_Report_Header --
   -------------------------

   procedure Print_Report_Header is
      Time_Of_Check   : constant Time := Clock;
      Month_Of_Check  : constant Month_Number := Month (Time_Of_Check);
      Day_Of_Check    : constant Day_Number   := Day (Time_Of_Check);
      Sec_Of_Check    : constant Day_Duration := Seconds (Time_Of_Check);

      Hour_Of_Chech   :          Integer range 0 .. 23;
      Minute_Of_Check :          Integer range 0 .. 59;
      Seconds_In_Hour : constant Integer := 60 * 60;

   begin
      Report ("GNATCheck report");
      Report_EOL;

      Report_No_EOL ("date              : ");
      Report_No_EOL (Trim (Year (Time_Of_Check)'Img, Left) & '-');

      if Month_Of_Check < 10 then
         Report_No_EOL ("0");
      end if;

      Report_No_EOL (Trim (Month_Of_Check'Img, Left) & '-');

      if Day_Of_Check < 10 then
         Report_No_EOL ("0");
      end if;

      Report_No_EOL (Trim (Day_Of_Check'Img, Left) & ' ');

      Hour_Of_Chech   := Integer (Sec_Of_Check) / Seconds_In_Hour;
      Minute_Of_Check := (Integer (Sec_Of_Check) rem Seconds_In_Hour) / 60;

      if Hour_Of_Chech < 10 then
         Report_No_EOL ("0");
      end if;

      Report_No_EOL (Trim (Hour_Of_Chech'Img, Left) & ':');

      if Minute_Of_Check < 10 then
         Report_No_EOL ("0");
      end if;

      Report        (Trim (Minute_Of_Check'Img, Left));

      Report_No_EOL ("gnatcheck version : ");
      Report_No_EOL (Tool_Name.all &  ' ');
      Report        (Gnat_Version_String);

      Report_No_EOL ("command line      : ");
      Print_Gnatcheck_Command_Line;

      Report_No_EOL ("runtime           : ");
      Print_Runtime;
   end Print_Report_Header;

   -------------------
   -- Print_Runtime --
   -------------------

   procedure Print_Runtime is
   begin
      if ASIS_UL.Compiler_Options.Custom_RTS /= null then
         Report (ASIS_UL.Compiler_Options.Custom_RTS.all);
      else
         Report ("<default>");
      end if;
   end Print_Runtime;

   ---------------------
   -- Print_Section_1 --
   ---------------------

   procedure Print_Section_1 (Exempted_Rules : Boolean := False) is
      SF_Name : String_Access;
      --  Points to the name of the current source file. We use a string access
      --  value instead of using ASIS_UL.Source_Table.Short_Source_Name because
      --  of the performance reasons

      Next_Diag : Diag_Id;

      Line_Pos : Natural := 0;
      Col_Pos  : Natural := 0;
      --  Max positions for lines and columns, are  needed to allign
      --  diagnoses in the report file

      Diagnoses_Reported : Natural := 0;
      --  Counts diagnoses that are prnted out into Stdout
   begin

      if Output_Section_1
        and then
         not Short_Report
      then
         Report_EOL;
         Report ("-------- Start Section 1 ------------");
         Report ("   (compiler-style report, diagnoses are grouped by files,");
         Report ("    and for each file diagnoses are ordered by increasing");
         Report ("    the source location of the corresponding construct)");
         Report_EOL;
      end if;

      for SF in First_SF_Id .. Last_Source loop
         Next_Diag := First_Diagnosis (SF, Exempted_Rules);

         if Present (Next_Diag) then

            SF_Name := new String'(Short_Source_Name (SF));

            if Output_Section_1
              and then
               not Full_Source_Locations
              and then
               not No_Column_Num_In_Diagnoses
            then
               Compute_Alignment (SF, Exempted_Rules, Line_Pos, Col_Pos);
            end if;

            while Present (Next_Diag) loop

               if Output_Section_1 then

                  if Full_Source_Locations
                     and then Diag_Table (Next_Diag).Rule not in
                              Warning_Diags .. Restriction_Diags
                  then
                     Report_No_EOL
                       (Get_String (Diag_Table (Next_Diag).SLOC));
                  else
                     Report_No_EOL (SF_Name.all & ':');

                     if No_Column_Num_In_Diagnoses then
                        Report_No_EOL
                          (Image (Integer (Diag_Table (Next_Diag).Line)));
                     else
                        Report_No_EOL
                          (Line_Col
                            (D        => Next_Diag,
                             Line_Pos => Line_Pos,
                             Col_Pos  => Col_Pos));
                     end if;

                  end if;

                  Report_No_EOL (": " & Text_Diag (Next_Diag));

                  if Exempted_Rules then
                     pragma Assert (Is_For_Exempted_Rule (Next_Diag));

                     Report (" (" & Text_Justification (Next_Diag) & ")");
                  else
                     Report_EOL;
                  end if;

               end if;

               --  This generates the diagnostic messages into Stdout
               if not Quiet_Mode
                 and then
                   (not Exemption_On
                   or else
                    not Is_For_Exempted_Rule (Next_Diag))
                 and then
                   (Max_Diagnoses = 0
                    or else
                     Max_Diagnoses > Diagnoses_Reported)
               then

                  if Diag_Table (Next_Diag).Rule not in
                     Warning_Diags .. Restriction_Diags
                  then
                     Put_Line
                       (Strip_Column
                         (Get_String (Diag_Table (Next_Diag).SLOC)) &
                          ": " &
                          Text_Diag (Next_Diag));
                  else
                     Put (SF_Name.all & ':');
                     Put (Image (Integer (Diag_Table (Next_Diag).Line)) &
                          ':');

                     if not No_Column_Num_In_Diagnoses then
                        Put (Image (Integer (Diag_Table (Next_Diag).Col))
                             & ":");
                     end if;

                     Put_Line  (' ' & Text_Diag (Next_Diag));
                  end if;

                  Diagnoses_Reported := Diagnoses_Reported + 1;

                  if Diagnoses_Reported = Max_Diagnoses then
                     Error ("Maximum diagnoses reached, "&
                            "see the report file for full details");
                  end if;

               end if;

               Next_Diag := Next_Diagnosis (Next_Diag, Exempted_Rules);
            end loop;

            Free (SF_Name);
         end if;

      end loop;

      if not Short_Report then
         Report ("-------- End Section 1 ------------");
         Report_EOL;
      end if;

   end Print_Section_1;

   ---------------------
   -- Print_Section_2 --
   ---------------------

   procedure Print_Section_2 (Exempted_Rules : Boolean := False) is

      procedure Print_Rule_Matches (Rule : Rule_Id);
      --  Prints out matches for a given rule detected in all the files
      --  that have been processed

      procedure Print_Rule_Matches (Rule : Rule_Id) is
         SF_Name : String_Access;
         --  Points to the name of the current source file. We use a string
         --  access value instead of using
         --  ASIS_UL.Source_Table.Short_Source_Name because of the performance
         --  reasons

         Next_Diag  : Diag_Id;
         No_Matches : Boolean := True;

      begin
         for SF in First_SF_Id .. Last_Source loop
            SF_Name   := new String'(Short_Source_Name (SF));
            Next_Diag := First_Rule_Diagnosis (SF, Rule, Exempted_Rules);

            if Present (Next_Diag) then
               Report ("Matches detected in file " & SF_Name.all, 1);

               while Present (Next_Diag) loop

                  if Full_Source_Locations then
                     Report (Get_String (Diag_Table (Next_Diag).SLOC), 2);
                  else
                     Report_No_EOL (Line_Col (Next_Diag), 2);

                     if Exempted_Rules then
                        pragma Assert (Is_For_Exempted_Rule (Next_Diag));

                        Report (" (" & Text_Justification (Next_Diag) & ")");
                     else
                        Report_EOL;
                     end if;

                  end if;

                  Next_Diag :=
                    Next_Same_Rule_Diagnosis (Next_Diag, Exempted_Rules);
               end loop;

               Report_EOL;

               No_Matches := False;

            end if;

            --  Report_EOL;
            Free (SF_Name);
         end loop;

         if No_Matches then
               Report ("No matches detected in processed files", 2);
         end if;

      end Print_Rule_Matches;

   begin
      if not Short_Report then
         Report_EOL;
         Report ("-------- Start Section 2 ------------");
         Report ("   (diagnoses are grouped by rules, and for each rule -");
         Report ("    by files, and for each file - by increasing the source");
         Report ("    location of the corresponding construct)");
         Report_EOL;
      end if;

      for Rule in All_Rules.First .. All_Rules.Last loop

         if All_Rules.Table (Rule).Rule_State /= Disabled
           and then
            All_Rules.Table (Rule).Diagnosis /= null
         then

            --  Note, that if a rule does not have its own diagnoses, this
            --  means that it is implemented by some other rules, so it
            --  should not go into the report

            --  Rule identification info, something more smart should be
            --  printed out here  ???

            Report (All_Rules.Table (Rule).Help_Info.all);
            Print_Rule_Matches (Rule);
            Report_EOL;
         end if;

      end loop;

      --  Compiler-made checks:

      if Use_gnaty_Option then
         Report ("Compiler style checks");
         Print_Rule_Matches (Style_Diags);
      end if;

      if Use_gnatw_Option then
         Report ("Compiler warnings");
         Print_Rule_Matches (Warning_Diags);
      end if;

      if Check_Restrictions then
         Report ("Compiler-checked restrictions");
         Print_Rule_Matches (Restriction_Diags);
      end if;

      if not Short_Report then
         Report ("-------- End Section 2 ------------");
         Report_EOL;
      end if;

   end Print_Section_2;

   ---------------------
   -- Print_Section_3 --
   ---------------------

   procedure Print_Section_3 (Exempted_Rules : Boolean := False) is
      SF_Name : String_Access;
      --  Points to the name of the current source file. We use a string access
      --  value instead of using ASIS_UL.Source_Table.Short_Source_Name because
      --  of the performance reasons

      Next_Diag : Diag_Id;

      procedure Print_Rule_Matches (Rule : Rule_Id; SF : SF_Id);
      --  Prints out matches for a given rule detected in the given file

      procedure Print_Rule_Matches (Rule : Rule_Id; SF : SF_Id) is
         Next_Diag : Diag_Id;
      begin
         Next_Diag := First_Rule_Diagnosis (SF, Rule, Exempted_Rules);

         if Present (Next_Diag) then

            while Present (Next_Diag) loop

               if Full_Source_Locations then
                  Report (Get_String (Diag_Table (Next_Diag).SLOC), 2);
               else
                  Report_No_EOL (Line_Col (Next_Diag), 2);

                  if Exempted_Rules then
                     pragma Assert (Is_For_Exempted_Rule (Next_Diag));

                     Report (" (" & Text_Justification (Next_Diag) & ")");
                  else
                     Report_EOL;
                  end if;

               end if;

               Next_Diag :=
                 Next_Same_Rule_Diagnosis (Next_Diag, Exempted_Rules);
            end loop;

         else
            Report ("No matches detected", 2);
         end if;

      end Print_Rule_Matches;

   begin
      if not Short_Report then
         Report_EOL;
         Report ("-------- Start Section 3 ------------");
         Report ("   (diagnoses are grouped by files, and for each file they");
         Report ("    are first grouped by rules and then - by increasing");
         Report ("    the source location of the corresponding construct)");
         Report_EOL;
      end if;

--      for SF in First_SF_Id .. Last_Argument_Source loop
      for SF in First_SF_Id .. Last_Source loop
         Next_Diag := First_Diagnosis (SF, Exempted_Rules);
         SF_Name   := new String'(Short_Source_Name (SF));

         if Present (Next_Diag) then

            Report ("Matches detected in file " & SF_Name.all);

            for Rule in All_Rules.First .. All_Rules.Last loop

               if All_Rules.Table (Rule).Rule_State /= Disabled
                 and then
                  All_Rules.Table (Rule).Diagnosis /= null
               then

                  --  Note, that if a rule does not have its own diagnoses,
                  --  this means that it is implemented by some other rules, so
                  --  it  should not go into the report

                  --  Rule identification info, something more smart should be
                  --  printed out here  ???

                  Report (All_Rules.Table (Rule).Help_Info.all, 1);
                  Print_Rule_Matches (Rule, SF);
                  Report_EOL;
               end if;

            end loop;

            Report_EOL;

            --  Compiler-made checks:

            if Use_gnaty_Option then
               Report ("Compiler style checks");
               Print_Rule_Matches (Style_Diags, SF);
            end if;

            if Use_gnatw_Option then
               Report ("Compiler warnings");
               Print_Rule_Matches (Warning_Diags, SF);
            end if;

            if Check_Restrictions then
               Report ("Compiler-checked restrictions");
               Print_Rule_Matches (Restriction_Diags, SF);
            end if;

         elsif Source_Status (SF) = Processed then
            Report ("No matches for enabled rules detected in file " &
                     SF_Name.all);
            Report_EOL;
         end if;

         Free (SF_Name);

      end loop;

      if not Short_Report then
         Report ("-------- End Section 3 ------------");
      end if;

   end Print_Section_3;

   ----------------------------
   -- Print_Sections_Summary --
   ----------------------------

   procedure Print_Sections_Summary is
   begin
      Report ("This report contains following sections:");
      Report_EOL;

      Report_No_EOL ("Section 1 - ");

      if Output_Section_1 then
         Report ("compiler-style report, diagnoses are grouped by files and");
         Report ("            for each file ordered by line numbers");

      else
         Report ("skipped");
      end if;

      Report_No_EOL ("Section 2 - ");

      if Output_Section_2 then
         Report ("diagnoses are grouped by rules, then - by files and then -");
         Report ("            by line numbers");
      else
         Report ("skipped");
      end if;

      Report_No_EOL ("Section 3 - ");

      if Output_Section_3 then
         Report ("diagnoses are grouped by files, then - by rules and then -");
         Report ("            by line numbers");
      else
         Report ("skipped");
      end if;

   end Print_Sections_Summary;

   -----------------------
   -- Print_Source_List --
   -----------------------

   procedure Print_Source_List is
   begin
      Report_EOL;

      Report ("Checked argument sources:");

      for SF in First_SF_Id .. Last_Argument_Source loop
         Report_No_EOL (Short_Source_Name (SF), 1);

         if Source_Status (SF) = Not_A_Legal_Source then
            Report (" - illegal source, no check is made");
         else
            Report_EOL;
         end if;

      end loop;

      if ASIS_UL.Options.Buld_Call_Graph
        and then
         Last_Argument_Source < Last_Source
      then
         Report ("Additional sources analyzed to check global rules:");

         for SF in Last_Argument_Source + 1 .. Last_Source loop
            Report_No_EOL (Short_Source_Name (SF), 1);

            if Source_Status (SF) = Not_A_Legal_Source then
               Report (" - illegal source, no check is made");
            else
               Report_EOL;
            end if;

         end loop;
      end if;

      Report_EOL;

   end Print_Source_List;

   -------------------------------
   -- Print_Specified_Diagnoses --
   -------------------------------

   procedure Print_Specified_Diagnoses
     (Position : All_Error_Messages_Storage.Cursor)
   is
   begin
      if Diagnoses_To_Print
           (All_Error_Messages_Storage.Element (Position).Diagnosis_Kind)
      then
         Report (All_Error_Messages_Storage.Element (Position).Text.all);

         if All_Error_Messages_Storage.Element (Position).Justification /=
            null
         then
            Report
              ("(" &
               All_Error_Messages_Storage.Element (Position).Justification.all
               & ")",
               1);
         end if;
      end if;
   end Print_Specified_Diagnoses;

   ----------------------
   -- Print_Violations --
   ----------------------

   procedure Print_Violations (Exempted : Boolean) is
      Next_Diag : Diag_Id;

      Diagnoses_Reported : Natural := 0;
      --  Counts diagnoses that are prnted out into Stdout

      Nothing_To_Report : Boolean := True;
   begin

      --  First, check if we have anything to report
      for SF in First_SF_Id .. Last_Source loop
         Next_Diag := First_Diagnosis (SF, Exempted);

         if Present (Next_Diag) then
            Nothing_To_Report := False;
            exit;
         end if;

      end loop;

      if Nothing_To_Report then
         Report_No_EOL ("No ");

         if not Exempted then
            Report_No_EOL ("non-");
         end if;

         Report ("exempted violations have been detected");
         if not Short_Report then
            Report_EOL;
         end if;

         return;
      end if;

      --  Printing out the list of files that contain violations

--      if not Short_Report then
--         Report_No_EOL ("Source files with ");

--         if not Exempted then
--            Report_No_EOL ("non-");
--         end if;

--         Report ("exempted violations");

--         for SF in First_SF_Id .. Last_Source loop
--            Next_Diag := First_Diagnosis (SF, Exempted);

--            if Present (Next_Diag) then
--               Report (Short_Source_Name (SF), 1);
--            end if;

--         end loop;

--         Report_EOL;

--         Report ("List of violations grouped by files, and ordered by " &
--                 "increasing source location:");
--         Report_EOL;
--      end if;

      --  Printing out individual diagnoses:

      for SF in First_SF_Id .. Last_Source loop
         Next_Diag := First_Diagnosis (SF, Exempted);

         if Present (Next_Diag) then

            while Present (Next_Diag) loop
               if Diag_Table (Next_Diag).Rule not in
                  Warning_Diags .. Restriction_Diags
               then
                  Report_No_EOL (Get_String (Diag_Table (Next_Diag).SLOC));
               else
                  Report_No_EOL (Short_Source_Name (SF) & ':');
                  Report_No_EOL
                    (Image (Integer (Diag_Table (Next_Diag).Line)) & ':');
                  Report_No_EOL
                    (Image (Integer (Diag_Table (Next_Diag).Col)));
               end if;

               Report (": " & Text_Diag (Next_Diag));

               if Exempted then
                  pragma Assert (Is_For_Exempted_Rule (Next_Diag));

                  Report ("(" & Text_Justification (Next_Diag) & ")", 1);
               end if;

               --  This generates the diagnostic messages into Stdout
               if not Quiet_Mode
                 and then
                   not Exempted
                 and then
                   (Max_Diagnoses = 0
                    or else
                     Max_Diagnoses > Diagnoses_Reported)
               then
                  if Diag_Table (Next_Diag).Rule not in
                     Warning_Diags .. Restriction_Diags
                  then
                     Put (Get_String (Diag_Table (Next_Diag).SLOC)  & ": ");
                  else
                     Put (Short_Source_Name (SF) & ':');
                     Put (Image (Integer (Diag_Table (Next_Diag).Line)) & ':');
                     Put (Image (Integer (Diag_Table (Next_Diag).Col)) & ": ");
                  end if;

                  Put_Line (Text_Diag (Next_Diag));

                  Diagnoses_Reported := Diagnoses_Reported + 1;

                  if Diagnoses_Reported = Max_Diagnoses then
                     Error ("Maximum diagnoses reached, "&
                            "see the report file for full details");
                  end if;

               end if;

               Next_Diag := Next_Diagnosis (Next_Diag, Exempted);
            end loop;

         end if;

      end loop;

      if not Short_Report then
         Report_EOL;
      end if;
   end Print_Violations;

   -----------------------------
   -- Print_Violation_Summary --
   -----------------------------

   procedure Print_Violation_Summary is
      Tmp : Natural;
   begin
      Tmp :=
        Error_Statistics (Non_Exempted_Rule_Violation) +
        Error_Statistics (Non_Exempted_Compiler_Check_Violation);

      Report
        ("non-exempted violations               :" & Tmp'Img, 1);

      Report
        ("rule exemption warnings               :" &
         Error_Statistics (Exemption_Warning)'Img, 1);

      Report
        ("compilation errors                    :" &
         Error_Statistics (Compiler_Error)'Img, 1);

      Report
        ("exempted violations                   :" &
         Error_Statistics (Exempted_Rule_Violation)'Img, 1);
   end Print_Violation_Summary;

   ------------------------------------------
   -- Process_Report_File_Format_Parameter --
   ------------------------------------------
   procedure Process_Report_File_Format_Parameter
     (Parameter :     String;
      Success   : out Boolean)
   is
   begin
      Success := True;

      if Parameter = "" then
         Short_Report := True;
         return;
      end if;

      if All_Section_On then
         All_Section_On  := False;
         Output_Section_1 := False;
         Output_Section_2 := False;
         Output_Section_3 := False;
      end if;

      for J in Parameter'Range loop

         case Parameter (J) is
            when '1' =>
               Output_Section_1 := True;
            when '2' =>
               Output_Section_2 := True;
            when '3' =>
               Output_Section_3 := True;
            when others =>
               Success := False;
               Error ("Wrong parameter of '-s' option: " & Parameter);
               return;
         end case;

      end loop;

   end Process_Report_File_Format_Parameter;

   ---------------------------
   -- Process_User_Filename --
   ---------------------------

   procedure Process_User_Filename (Fname : String) is
   begin

      if Is_Regular_File (Fname) then

         if User_Info_File /= null then
            Error ("--include-file option can be given only once, " &
                   "all but first ignored");
         else
            User_Info_File           := new String'(Fname);
            User_Info_File_Full_Path := new String'
              (Normalize_Pathname
                 (Fname,
                  Resolve_Links  => False,
                  Case_Sensitive => False));
         end if;

      else
         Error (Fname & " not found, --include-file option ignored");
      end if;

   end Process_User_Filename;

   -------------------------
   -- Set_First_Diagnosis --
   -------------------------

   procedure Set_First_Diagnosis (SF : SF_Id; D : Diag_Id) is
   begin
      Mapping_Table (SF) (All_Diags).First := D;
   end Set_First_Diagnosis;

   ------------------------------
   -- Set_First_Rule_Diagnosis --
   ------------------------------

   procedure Set_First_Rule_Diagnosis (SF : SF_Id; R : Rule_Id; D : Diag_Id) is
   begin
      Mapping_Table (SF) (R).First := D;
   end Set_First_Rule_Diagnosis;

   ------------------------
   -- Set_Last_Diagnosis --
   ------------------------

   procedure Set_Last_Diagnosis (SF : SF_Id; D : Diag_Id) is
   begin
      Mapping_Table (SF) (All_Diags).Last := D;
   end Set_Last_Diagnosis;

   -----------------------------
   -- Set_Last_Rule_Diagnosis --
   -----------------------------

   procedure Set_Last_Rule_Diagnosis (SF : SF_Id; R : Rule_Id; D : Diag_Id) is
   begin
      Mapping_Table (SF) (R).Last := D;
   end Set_Last_Rule_Diagnosis;

   ----------------------------
   -- Store_Compiler_Message --
   ----------------------------

   procedure Store_Compiler_Message
     (In_SF        : SF_Id;
      Line_Num     : Natural;
      Col_Num      : Natural;
      Message      : String_Loc;
      Message_Kind : Compiler_Message_Kinds)
   is
      For_Rule : Rule_Id;
      --  Artificial Rule_Id for storing the compiler message
   begin
      case Message_Kind is
         when Not_A_Compiler_Nessage =>
            pragma Assert (False);
            return;
         when General_Warning =>
            For_Rule := Warning_Diags;
         when Style =>
            For_Rule := Style_Diags;
         when Restriction =>
            For_Rule := Restriction_Diags;
      end case;

      Store_Rule_Violation_Internal
        (For_Rule      => For_Rule,
         Line_Num      => Types.Physical_Line_Number (Line_Num),
         Col_Num       => Types.Column_Number (Col_Num),
         In_SF         => In_SF,
         Justification => Nil_String_Loc, --  exemption for compiler tests is
                                          --  not implemented yet
         Diagnosis_Num => 0,   --  temporary solution
         Diag_Text     => Message,
         Element_SLOC  => Nil_String_Loc);
   end Store_Compiler_Message;

   ----------------------------
   -- Store_Compiler_Warning --
   ----------------------------

   procedure Store_Compiler_Warning
     (Text : String;
      SF   : SF_Id)
   is
   begin
      Store_Diagnosis
        (Text           => Text,
         Diagnosis_Kind => Non_Exempted_Compiler_Check_Violation,
         SF             => SF);
   end Store_Compiler_Warning;

   ---------------------
   -- Store_Diagnosis --
   ---------------------

   procedure Store_Diagnosis
     (Text           : String;
      Diagnosis_Kind : Diagnosis_Kinds;
      SF             : SF_Id;
      Justification  : String := "")
   is
   begin
      All_Error_Messages_Storage.Insert
        (Container => All_Error_Messages,
         New_Item  => (Text           => new String'(Text),
                       Justification  => Get_Justification (Justification),
                       Diagnosis_Kind => Diagnosis_Kind,
                       SF             => SF,
                       Num            => Next_Message_Num),
         Position  => Unused_Position,
         Inserted  => Unused_Inserted);
   end Store_Diagnosis;

   --------------------------
   -- Store_Error_Messages --
   --------------------------

   procedure Store_Error_Messages
     (Compiler_Out_Fname : String;
      SF                 : SF_Id)
   is
      Comp_Out_File : Ada.Text_IO.File_Type;
      Line_Buffer   : String (1 .. 16 * 1024);
      Line_Len      : Natural := 0;

      Is_Error_Massage    : Boolean;
      Error_Message_Found : Boolean := False;
   begin
      pragma Assert (Source_Status (SF) = Not_A_Legal_Source);

      if not Is_Regular_File (Compiler_Out_Fname) then
         Error ("no compiler message file found for " & Source_Name (SF));
         return;
      end if;

      Open
        (File => Comp_Out_File,
         Mode => In_File,
         Name => Compiler_Out_Fname);

      while not End_Of_File (Comp_Out_File) loop
         Get_Line
           (File => Comp_Out_File,
            Item => Line_Buffer,
            Last => Line_Len);

         Is_Error_Massage :=
           Index
             (Source  => Line_Buffer (1 .. Line_Len),
              Pattern => "(style)") = 0;

         if Is_Error_Massage then
            Is_Error_Massage :=
              Index
                (Source  => Line_Buffer (1 .. Line_Len),
                 Pattern => ": warning:") = 0;
         end if;

         if Is_Error_Massage then
            Error_Message_Found := True;

--            Error_No_Tool_Name (Line_Buffer (1 .. Line_Len));

            Store_Diagnosis
              (Text           => Line_Buffer (1 .. Line_Len),
               Diagnosis_Kind => Compiler_Error,
               SF             => SF);
         end if;

      end loop;

      if not Error_Message_Found then
--         Error_No_Tool_Name
--           (Short_Source_Name (SF) &
--            ": cannot be compiled by unknown reason");

         Store_Diagnosis
           (Text           => Short_Source_Name (SF) &
                              ":1:1: cannot be compiled by unknown reason",
            Diagnosis_Kind => Compiler_Error,
            SF             => SF);
      end if;

      Close (Comp_Out_File);
   end Store_Error_Messages;

   --------------------------
   -- Store_Rule_Violation --
   --------------------------

   procedure Store_Rule_Violation
     (For_Rule : Rule_Id;
      On       : GS_Node_Id)
   is
      --  Should We keep line and column numbers as a part of the global
      --  structure node structure

      SLOC : constant String := Get_String (GS_Node_SLOC (On));

      function Get_Line_Number return Types.Physical_Line_Number;
      --  Separates the line number from SLOC and returns it as the value
      --  of Physical_Line_Number

      function Get_Column_Number return Types.Column_Number;
      --  Separates the line number from SLOC and returns it as the value
      --  of Column_Number

      function Get_Line_Number return Types.Physical_Line_Number is
         First_Colon  : Positive;
         Second_Colon : Positive;
      begin
         First_Colon := Index (SLOC, ":");
         Second_Colon := Index (SLOC (First_Colon + 1 .. SLOC'Last), ":");

         return Types.Physical_Line_Number'Value
           (SLOC (First_Colon + 1 .. Second_Colon - 1));
      end Get_Line_Number;

      function Get_Column_Number return Types.Column_Number is
         Second_Colon : Positive;
         Third_Colon  : Natural;
      begin
         Second_Colon := Index (SLOC, ":");
         Second_Colon := Index (SLOC (Second_Colon + 1 .. SLOC'Last), ":");
         Third_Colon  := Index (SLOC (Second_Colon + 1 .. SLOC'Last), "[");

         if Third_Colon = 0 then
            Third_Colon := SLOC'Last;
         else
            Third_Colon := Third_Colon - 1;
         end if;

         return Types.Column_Number'Value
           (SLOC (Second_Colon + 1 .. Third_Colon));
      end Get_Column_Number;

      Line_Num : constant Types.Physical_Line_Number := Get_Line_Number;
      Col_Num  : constant Types.Column_Number        := Get_Column_Number;

   begin

      Store_Rule_Violation_Internal
        (For_Rule      => For_Rule,
         Line_Num      => Line_Num,
         Col_Num       => Col_Num,
         In_SF         => Enclosing_Source (On),
         Justification => Nil_String_Loc, --  exemption for global rules is
                                          --  not implemented yet
         Diagnosis_Num => 0, --  temporary solution
         Element_SLOC  => GS_Node_SLOC (On));

   end Store_Rule_Violation;

   procedure Store_Rule_Violation
     (For_Rule      : Rule_Id;
      On            : Element;
      In_SF         : SF_Id;
      Justification : String_Loc;
      Diagnosis_Num : Diagnosis_Variant := 0;
      Diag_Pars     : String_Loc;
      Element_SLOC  : String_Loc)
   is
      P        : constant Types.Source_Ptr := Sloc (Node (On));
      Line_Num : constant Types.Physical_Line_Number :=
        Get_Physical_Line_Number (P);
      Col_Num  : constant Types.Column_Number := Get_Column_Number (P);

   begin
      Store_Rule_Violation_Internal
        (For_Rule      => For_Rule,
         Line_Num      => Line_Num,
         Col_Num       => Col_Num,
         In_SF         => In_SF,
         Justification => Justification,
         Diagnosis_Num => Diagnosis_Num,
         Diag_Text     => Diag_Pars,
         Element_SLOC  => Element_SLOC);

   end Store_Rule_Violation;

   -----------------------------------
   -- Store_Rule_Violation_Internal --
   -----------------------------------

   procedure Store_Rule_Violation_Internal
     (For_Rule      : Rule_Id;
      Line_Num      : Types.Physical_Line_Number;
      Col_Num       : Types.Column_Number;
      In_SF         : SF_Id;
      Justification : String_Loc;
      Diagnosis_Num : Diagnosis_Variant := 0;
      Diag_Text     : String_Loc        := Nil_String_Loc;
      Element_SLOC  : String_Loc        := Nil_String_Loc)
   is
      New_Diag  : Diag_Id;
      Diag_Link : Diag_Id;
      Tmp       : Diag_Id;
      use Types;
      --  To make type operations visible in the body

      Duplication_Possible : Boolean := False;
   begin

      --  Check for possible duplications of diagnoses for compiler checks

      if For_Rule in Warning_Diags .. Restriction_Diags then
         Diag_Link := First_Rule_Diagnosis (In_SF, For_Rule);

         while Present (Diag_Link) loop
            --  Go to the diagnoses with same line and column
            if Line (Diag_Link) > Line_Num
             or else
               (Line (Diag_Link) = Line_Num
               and then
                Col (Diag_Link) > Col_Num)
            then
               exit;
            elsif Line (Diag_Link) = Line_Num
                and then
                  Col (Diag_Link) = Col_Num
            then
               Duplication_Possible := True;
               exit;
            else
               Diag_Link := Diag_Table (Diag_Link).Next_Same_Rule_Diag;
            end if;
         end loop;

         if Duplication_Possible then
            while Present (Diag_Link)
               and then
                  Line (Diag_Link) = Line_Num
                and then
                  Col (Diag_Link) = Col_Num
            loop

               if Text_Diag (Diag_Link) = Get_String (Diag_Text) then
                  --  Here we have a duplication, so - nothing to store!
                  return;
               else
                  Diag_Link := Diag_Table (Diag_Link).Next_Same_Rule_Diag;
               end if;

            end loop;
         end if;

      end if;

      if Justification /= Nil_String_Loc then
         --  Counting violations detected for an exempted rule
         Add_Exempted_Violation (For_Rule);
      end if;

      --  ??? Do we need to store the source file Id here?

      Rule_Violations.Append
        ((Rule                 => For_Rule,
          SF                   => In_SF,
          Diagnosis_Num        => Diagnosis_Num,
          Line                 => Line_Num,
          Col                  => Col_Num,
          Next_Diag            => No_Diag,
          Prev_Diag            => No_Diag,
          Next_Same_Rule_Diag  => No_Diag,
          Prev_Same_Rule_Diag  => No_Diag,
          Diag_Text            => Diag_Text,
          Exempt_Justification => Justification,
          SLOC                 => Element_SLOC));

      New_Diag := Rule_Violations.Last;

      --  And now we have to update the chains:

      Diag_Link := Last_Diagnosis (In_SF);

      if No (Diag_Link) then
         --  The first diagnosis for the given source

         Set_First_Diagnosis (In_SF, New_Diag);

         --  If this is the first diagnosis for the given source, it
         --  is the first diagnosis for the given rule in the given source
         --  as well.

         Set_First_Rule_Diagnosis (In_SF, For_Rule, New_Diag);

         Set_Last_Diagnosis (In_SF, New_Diag);
         Set_Last_Rule_Diagnosis (In_SF, For_Rule, New_Diag);
      else

         --  Set all diagnoses chain
         if For_Rule > All_Rules.Last
          or else
            All_Rules.Table (For_Rule).all in Global_Rule_Template'Class
         then
            Diag_Link := First_Diagnosis (In_SF);

            while Present (Diag_Link)
              and then
                 (Line (Diag_Link) < Line (New_Diag)
                 or else
                  (Line (Diag_Link) = Line (New_Diag)
                  and then
                   Col (Diag_Link) < Col (New_Diag)))
            loop
               Diag_Link := Diag_Table (Diag_Link).Next_Diag;
            end loop;

            if No (Diag_Link) then
               --  So the new diagnosis is the rightmost in this file
               Diag_Link                        := Last_Diagnosis (In_SF);
               Diag_Table (New_Diag).Prev_Diag  := Diag_Link;
               Diag_Table (Diag_Link).Next_Diag := New_Diag;
               Set_Last_Diagnosis (In_SF, New_Diag);
            else
               Tmp := Diag_Table (Diag_Link).Prev_Diag;
               --  We have in insert the new diagnosis between Tmp and
               --  Diag_Link

               if No (Tmp) then
                  --  The new diagnosis is the leftmost for this file
                  Set_First_Diagnosis (In_SF, New_Diag);

                  Diag_Table (New_Diag).Next_Diag  := Diag_Link;
                  Diag_Table (Diag_Link).Prev_Diag := New_Diag;
               else
                  Diag_Table (Tmp).Next_Diag  := New_Diag;

                  Diag_Table (New_Diag).Prev_Diag := Tmp;
                  Diag_Table (New_Diag).Next_Diag := Diag_Link;

                  Diag_Table (Diag_Link).Prev_Diag  := New_Diag;

               end if;

            end if;

         else
            --  Diag_Link points to the last diagnosis!
            Diag_Table (New_Diag).Prev_Diag  := Diag_Link;
            Diag_Table (Diag_Link).Next_Diag := New_Diag;
            Set_Last_Diagnosis (In_SF, New_Diag);
         end if;

         --  Update the rule chain

         if For_Rule > All_Rules.Last
           or else
            All_Rules.Table (For_Rule).all in Global_Rule_Template'Class
         then
            Diag_Link := First_Rule_Diagnosis (In_SF, For_Rule);

            if No (Diag_Link) then
               --  First diagnosis for the given rule
               Set_First_Rule_Diagnosis (In_SF, For_Rule, New_Diag);
               Set_Last_Rule_Diagnosis  (In_SF, For_Rule, New_Diag);
            else

               while Present (Diag_Link)
                 and then
                     (Line (Diag_Link) < Line (New_Diag)
                     or else
                      (Line (Diag_Link) = Line (New_Diag)
                      and then
                       Col (Diag_Link) < Col (New_Diag)))
               loop
                  Diag_Link := Diag_Table (Diag_Link).Next_Same_Rule_Diag;
               end loop;

               if No (Diag_Link) then
                  --  So the new diagnosis is the rightmost in this file
                  --  for the given rule

                  Diag_Link := Last_Rule_Diagnosis (In_SF, For_Rule);
                  Diag_Table (New_Diag).Prev_Same_Rule_Diag  := Diag_Link;
                  Diag_Table (Diag_Link).Next_Same_Rule_Diag := New_Diag;
                  Set_Last_Rule_Diagnosis (In_SF, For_Rule, New_Diag);
               else
                  Tmp := Diag_Table (Diag_Link).Prev_Same_Rule_Diag;
                  --  We have in insert the new diagnosis between Tmp and
                  --  Diag_Link

                  if No (Tmp) then
                     --  The new diagnosis is the leftmost for this file
                     Set_First_Rule_Diagnosis    (In_SF, For_Rule, New_Diag);

                     Diag_Table (New_Diag).Next_Same_Rule_Diag  := Diag_Link;
                     Diag_Table (Diag_Link).Prev_Same_Rule_Diag := New_Diag;
                  else
                     Diag_Table (Tmp).Next_Same_Rule_Diag := New_Diag;

                     Diag_Table (New_Diag).Prev_Same_Rule_Diag := Tmp;
                     Diag_Table (New_Diag).Next_Same_Rule_Diag := Diag_Link;

                     Diag_Table (Diag_Link).Prev_Same_Rule_Diag := New_Diag;

                  end if;

               end if;
            end if;

         else
            Diag_Link := Last_Rule_Diagnosis (In_SF, For_Rule);

            if No (Diag_Link) then
               --  The first diagnosis for the given rule in the given source
               Set_First_Rule_Diagnosis (In_SF, For_Rule, New_Diag);
            else
               Diag_Table (New_Diag).Prev_Same_Rule_Diag  := Diag_Link;
               Diag_Table (Diag_Link).Next_Same_Rule_Diag := New_Diag;
            end if;

            Set_Last_Rule_Diagnosis (In_SF, For_Rule, New_Diag);

         end if;

      end if;

      --  Update violation counters:
      Detected_Rule_Violations := Detected_Rule_Violations + 1;

      if Justification = Nil_String_Loc then
         Detected_Non_Exempted_Violations :=
           Detected_Non_Exempted_Violations + 1;
      else
         Detected_Exempted_Violations := Detected_Exempted_Violations + 1;
      end if;

   end Store_Rule_Violation_Internal;

   ------------------
   -- Strip_Column --
   ------------------

   function Strip_Column (SLOC : String) return String is
   begin

      if No_Column_Num_In_Diagnoses then
         return SLOC (SLOC'First .. Index (SLOC, ":", Backward) - 1);
      else
         return SLOC;
      end if;

   end Strip_Column;

   ---------------
   -- Text_Diag --
   ---------------

   function Text_Diag (D : Diag_Id) return String is
   begin

      if Diag_Table (D).Rule in Warning_Diags .. Restriction_Diags then
         return Get_String (Diag_Table (D).Diag_Text);
      else
         return Insert_Actuals
                  (Message => Select_Variant
                             (Message =>
                                 All_Rules.Table (Diag_Table (D).Rule).
                                    Diagnosis.all,
                              Num     => Diag_Table (D).Diagnosis_Num),

                   Actuals => Diag_Table (D).Diag_Text);
      end if;
   end Text_Diag;

   ------------------------
   -- Text_Justification --
   ------------------------

   function Text_Justification (D : Diag_Id) return String is
   begin
      return Get_String (Diag_Table (D).Exempt_Justification);
   end Text_Justification;

end Gnatcheck.Diagnoses;
