/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * rastrip - argus data client that strips unwanted DSR's from argus
 *           records. By default it strips them all.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <rastrip.h>

int ArgusStripDown = 0;
int RaFirstMOptionField = 1;


#define RASTRIP_DSR_TYPES                19

#define RASTRIP_DSR_FAR                  0
#define RASTRIP_DSR_MAC                  1
#define RASTRIP_DSR_TCP                  2
#define RASTRIP_DSR_ICMP                 3
#define RASTRIP_DSR_RTP                  4
#define RASTRIP_DSR_ARP                  6
#define RASTRIP_DSR_FRG                  7
#define RASTRIP_DSR_ESP                  8
#define RASTRIP_DSR_MPLS                 9
#define RASTRIP_DSR_VLAN                10
#define RASTRIP_DSR_PPPOE               11
#define RASTRIP_DSR_AGR                 12
#define RASTRIP_DSR_TIME                13
#define RASTRIP_DSR_SRCTIME             14
#define RASTRIP_DSR_DSTTIME             15 
#define RASTRIP_DSR_USRDATA		16      
#define RASTRIP_DSR_SRCUSRDATA		17      
#define RASTRIP_DSR_DSTUSRDATA		18      

int RaDSRFields[RASTRIP_DSR_TYPES];

char *RaDSRKeyWords[RASTRIP_DSR_TYPES] = {
   "far",
   "mac",
   "tcp",
   "icmp",
   "rtp",
   "igmp",
   "arp",
   "frag",
   "esp",
   "mpls",
   "vlan",
   "pppoe",
   "agr",
   "jitter",
   "stime",
   "dtime",
   "user",
   "srcuser",
   "dstuser",
};

void RaProcessMOptions(struct ArgusModeStruct *);

#define RA_ADD_OPTION		1
#define RA_SUB_OPTION		2

void
RaProcessMOptions(struct ArgusModeStruct *mode)
{
   int x, RaOptionOperation, setValue = 0;
   char *ptr = NULL;
   char *endptr;

   if (mode != NULL) {
      while (mode) {
         if (isdigit(*mode->mode)) {
             ArgusStripDown = strtol(mode->mode, &endptr, 10);
             if (mode->mode == endptr)
                usage();

         } else {
            if (*mode->mode == '-') {
               if (RaFirstMOptionField) {
                  for (x = 0; x < RASTRIP_DSR_TYPES; x++)
                     RaDSRFields[x] = 1;
                  RaFirstMOptionField = 0;
               }
               ptr = mode->mode + 1;
               RaOptionOperation = RA_SUB_OPTION;
            } else 
            if (*mode->mode == '+') {
               ptr = mode->mode + 1;
               RaOptionOperation = RA_ADD_OPTION;
            } else {
               if (RaFirstMOptionField) {
                  bzero ((char *) RaDSRFields, sizeof(RaDSRFields));
                  RaFirstMOptionField = 0;
               }
               ptr = mode->mode;
               RaOptionOperation = RA_ADD_OPTION;
            }

            setValue = (RaOptionOperation == RA_ADD_OPTION) ? 1 : 0;

            for (x = 0; x < RASTRIP_DSR_TYPES; x++) {
               if (!strncmp (RaDSRKeyWords[x], ptr, strlen(RaPrintKeyWords[x]))) {
                  switch (x) {
                     case RASTRIP_DSR_USRDATA: {
                        RaDSRFields[RASTRIP_DSR_SRCUSRDATA] = setValue;
                        RaDSRFields[RASTRIP_DSR_DSTUSRDATA] = setValue;
                        break;
                     }

                     default:
                        RaDSRFields[x] = setValue;
                        break;
                  }
               }
            }
            mode = mode->nxt;
         }
      }
   }
}

void
ArgusClientInit ()
{
   struct ArgusModeStruct *mode;

   if (!(RaInitialized)) {
      RaWriteOut = 0;
      RaCumulativeMerge = 1;

      if (ArgusInputFileList == NULL)
         usage();

      bzero ((char *)RaDSRFields, sizeof(RaDSRFields));
      RaDSRFields[RASTRIP_DSR_FAR] = 1;
      RaDSRFields[RASTRIP_DSR_TCP] = 1;

      if ((mode = ArgusModeList) != NULL)
         RaProcessMOptions(mode);

      if (!(ArgusStripDown))
         ArgusStripDown++;
   }
}


int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if ((sig >= 0) && (!RaParseCompleting))
      RaParseCompleting++;
}

void
ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{ 
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "Rastrip Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusProgramName);

   fprintf (stderr, "options: -b                dump packet-matching code.\n");
   fprintf (stderr, "         -C                treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -D <level>        specify debug level\n");
   fprintf (stderr, "         -F <conffile>     read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                print help.\n");
   fprintf (stderr, "         -r <file>         read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -t <timerange>    specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                           timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                               mm/dd[/yy]\n");
   fprintf (stderr, "                                               -%%d{yMhdms}\n");
   fprintf (stderr, "         -w <file>         write output to <file>. '-' denotes stdout.\n");
   exit(1);
}

int RaSendArgusRecord(struct ArgusRecordStore *argus) {return 0;}

#include <argus_util.h>

unsigned char argusBuf[2048];
extern struct ArgusCanonicalRecord *RaThisCanon;
struct ArgusRecord * RaConstructArgusRecord (struct ArgusRecord *);


struct ArgusRecord *
RaConstructArgusRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *newarg = NULL;
   struct ArgusFarHeaderStruct **hdrs = NULL, *hdr;
   
   if (argus != NULL) {
      ArgusThisFarStatus = ArgusIndexRecord (argus, ArgusThisFarHdrs);
      hdrs = ArgusThisFarHdrs;

      bzero (argusBuf, sizeof(argusBuf));
      newarg = (struct ArgusRecord *) argusBuf;

      bcopy ((char *)&argus->ahdr, (char *)&newarg->ahdr, sizeof (struct ArgusRecordHeader));
      newarg->ahdr.length = sizeof (struct ArgusRecordHeader);

      if (argus->ahdr.type & ARGUS_MAR) {
         bcopy ((char *)&argus->argus_mar, (char *)&newarg->argus_mar, sizeof(struct ArgusMarStruct));
         newarg->ahdr.length += sizeof(struct ArgusMarStruct);

      } else {
         if (((hdr = hdrs[ARGUS_FAR_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_FAR])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_MAC_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_MAC])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_TCP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_TCP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_VLAN_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_VLAN])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_MPLS_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_MPLS])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_AGR_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_AGR])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_TIME_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_TIME])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_ICMP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_ICMP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_RTP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_RTP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_ARP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_ARP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_FRG_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_FRG])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_ESP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_ESP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_SRCUSRDATA_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_SRCUSRDATA])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_DSTUSRDATA_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_DSTUSRDATA])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }
      }
   }

   return (newarg);
}

void RaGenerateCanonicalRecord (struct ArgusRecord *ptr, struct ArgusCanonicalRecord *);
int RaInitialRecord = 0;

void
RaProcessRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *retn = NULL;

   if (!(RaInitialRecord)) {
      RaInitialRecord++;

   } else {
      if ((retn = RaConstructArgusRecord (argus)) != NULL) {
         if ((ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusWfileList)))) {
            struct ArgusWfileStruct *wfile = NULL, *start = NULL;
 
            if ((wfile = ArgusFrontList(ArgusWfileList)) != NULL) { 
               start = wfile; 
               do { 
                  if ((exceptfile == NULL) || strcmp(wfile->filename, exceptfile)) { 
#ifdef _LITTLE_ENDIAN
                     ArgusHtoN(retn); 
#endif
                     ArgusWriteNewLogfile (wfile, retn); 
#ifdef _LITTLE_ENDIAN
                     ArgusNtoH(retn); 
#endif
                  }   
                  ArgusPopFrontList(ArgusWfileList); 
                  ArgusPushBackList(ArgusWfileList, wfile); 
                  wfile = ArgusFrontList(ArgusWfileList); 

               } while (wfile != start); 
            }   
         }
      }
   }
}


void
RaGenerateCanonicalRecord (struct ArgusRecord *ptr, struct ArgusCanonicalRecord *canon)
{
   int i, index = 0;
   struct ArgusFarHeaderStruct **hdrs = NULL;
   struct ArgusRecord *ar = (struct ArgusRecord *) canon;

   ArgusThisFarStatus = ArgusIndexRecord (ptr, ArgusThisFarHdrs);
   hdrs = ArgusThisFarHdrs;
   bzero ((char *)canon, sizeof(*canon));

   if (ptr->ahdr.type & ARGUS_MAR) {
      bcopy ((char *)ptr, (char *)canon, sizeof(*ptr));

   } else {
      bcopy ((char *)&ptr->ahdr, (char *)&canon->ahdr, sizeof(canon->ahdr));
   
      for (i = 1; i < 33; i++) {
         index = 1 << (i - 1);
         switch (index) {
            case ARGUS_FAR_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_FAR_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_FAR_DSR_INDEX], (char *)&ar->argus_far, sizeof (ar->argus_far));
               break;
            case ARGUS_MAC_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_MAC_DSR_INDEX], (char *)&canon->mac, sizeof(canon->mac));
               break;
            case ARGUS_VLAN_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_VLAN_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_VLAN_DSR_INDEX], (char *)&canon->vlan, sizeof(canon->vlan));
               break;
            case ARGUS_MPLS_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_MPLS_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_MPLS_DSR_INDEX], (char *)&canon->mpls, sizeof(canon->mpls));
               break;
            case ARGUS_AGR_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_AGR_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_AGR_DSR_INDEX], (char *)&canon->agr, sizeof(canon->agr));
               break;
            case ARGUS_TIME_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_TIME_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_TIME_DSR_INDEX], (char *)&canon->time, sizeof(canon->time));
               break;
            case ARGUS_TCP_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_TCP_DSR_INDEX], (char *)&canon->acr_tcp, sizeof(canon->acr_tcp));
               break;
            case ARGUS_ICMP_DSR_STATUS:  
               if (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_ICMP_DSR_INDEX], (char *)&canon->acr_icmp, sizeof(canon->acr_icmp));
               break;
            case ARGUS_RTP_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_RTP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_RTP_DSR_INDEX], (char *)&canon->acr_rtp, sizeof(canon->acr_rtp));
               break;
            case ARGUS_ARP_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_ARP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_ARP_DSR_INDEX], (char *)&canon->acr_arp, sizeof(canon->acr_arp));
               break;
            case ARGUS_FRG_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_FRG_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_FRG_DSR_INDEX], (char *)&canon->acr_frag, sizeof(canon->acr_frag));
               break;
            case ARGUS_ESP_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_ESP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_ESP_DSR_INDEX], (char *)&canon->acr_esp, sizeof(canon->acr_esp));
               break;
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "RaGenerateCanonicalRecord (0x%x, 0x%x) returning\n", ptr, canon);
#endif
}
