/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#pragma once

const char GetPMTDecl[] =
"double GetPMT( double fRate, double fNper, double fPv, double fFv, bool bPayInAdvance);\n";

const char GetPMT[] =
"double GetPMT( double fRate, double fNper, double fPv, double fFv, bool bPayInAdvance )\n"
"{\n"
"    double fPayment;\n"
"    if (fRate == 0.0)\n"
"        fPayment = (fPv + fFv) / fNper;\n"
"    else\n"
"    {\n"
"        if (bPayInAdvance)\n"
"            fPayment = (fFv + fPv * exp( fNper * log1p(fRate) ) ) * fRate\n"
"                / (expm1( (fNper + 1) * log1p(fRate) ) - fRate);\n"
"        else\n"
"            fPayment = (fFv + fPv * exp(fNper * log1p(fRate) ) ) * fRate\n"
"                / expm1( fNper * log1p(fRate) );\n"
"    }\n"
"    return -fPayment;\n"
"}\n";

const char GetIpmtDecl[] =
"double GetIpmt(double fRate, double fPer, double fNper, double fPv,\n"
"    double fFv, bool bPayInAdvance, double* fPmt);\n";

const char GetIpmt[] =
"double GetIpmt(double fRate, double fPer, double fNper, double fPv,\n"
"    double fFv, bool bPayInAdvance, double* fPmt)\n"
"{\n"
"    *fPmt = GetPMT(fRate, fNper, fPv, fFv, bPayInAdvance);\n"
"    double fIpmt;\n"
"    if (fPer == 1.0)\n"
"    {\n"
"        if (bPayInAdvance)\n"
"            fIpmt = 0.0;\n"
"        else\n"
"            fIpmt = -fPv;\n"
"    }\n"
"    else\n"
"    {\n"
"        if (bPayInAdvance)\n"
"            fIpmt = GetFV(fRate, fPer-2.0, *fPmt, fPv, true) - *fPmt;\n"
"        else\n"
"            fIpmt = GetFV(fRate, fPer-1.0, *fPmt, fPv, false);\n"
"    }\n"
"    return fIpmt * fRate;\n"
"}\n";

const char GetFVDecl[] =
"double GetFV( double fRate, double fNper, double fPmt,"
"    double fPv, bool bPayInAdvance );\n";

const char GetFV[] =
"double GetFV( double fRate, double fNper, double fPmt,"
"    double fPv, bool bPayInAdvance )\n"
"{\n"
"    double fFv;\n"
"    if (fRate == 0.0)\n"
"        fFv = fPv + fPmt * fNper;\n"
"    else\n"
"    {\n"
"        double fTerm = pow(1.0 + fRate, fNper);\n"
"        if (bPayInAdvance)\n"
"            fFv = fPv * fTerm + fPmt*(1.0 + fRate)*(fTerm - 1.0)/fRate;\n"
"        else\n"
"            fFv = fPv * fTerm + fPmt*(fTerm - 1.0)/fRate;\n"
"    }\n"
"    return -fFv;\n"
"}\n";

const char IsLeapYearDecl[] =
"bool IsLeapYear( int n );\n";

const char IsLeapYear[] =
"bool IsLeapYear( int n )\n"
"{\n"
"    return ( (( ( n % 4 ) == 0 ) && ( ( n % 100 ) != 0)) || ( ( n % 400 ) == "
"0 ) );\n"
"}\n";

const char DaysInMonthDecl[] =
"int DaysInMonth( int nMonth, int nYear );\n";

const char DaysInMonth[] =
"int DaysInMonth( int nMonth, int nYear )\n"
"{\n"
"    int tmp = 0;\n"
"    switch(nMonth)\n"
"    {\n"
"    case 1:\n"
"    case 3:\n"
"    case 5:\n"
"    case 7:\n"
"    case 8:\n"
"    case 10:\n"
"    case 12:\n"
"        tmp = 31;\n"
"        break;\n"
"    case 4:\n"
"    case 6:\n"
"    case 9:\n"
"    case 11:\n"
"        tmp =30;\n"
"        break;\n"
"    case 2:\n"
"        if ( IsLeapYear(nYear)==1)\n"
"            tmp = 29;\n"
"        else\n"
"            tmp = 28;\n"
"        break;\n"
"    }\n"
"    return tmp;\n"
"}\n";

const char DateToDaysDecl[] =
"int  DateToDays( int nDay, int nMonth, int nYear );\n";

const char DateToDays[] =
"int  DateToDays( int nDay, int nMonth, int nYear )\n"
"{\n"
"    int nDays = (nYear-1) * 365;\n"
"    nDays += ((nYear-1) / 4) - ((nYear-1) / 100) + ((nYear-1) / 400);\n"
"    for( int i = 1; i < nMonth; i++ )\n"
"        nDays += DaysInMonth(i,nYear);\n"
"    nDays += nDay;\n"
"\n"
"    return nDays;\n"
"}\n";

const char GetNullDateDecl[] =
"int GetNullDate();\n";

const char GetNullDate[] =
"int GetNullDate()\n"
"{\n"
"    return DateToDays(30,12,1899 );\n"
"}\n";

const char ScaDateDecl[] =
"void ScaDate( int nNullDate, int nDate, int nBase,int *nOrigDay, "
"int *nMonth,int *nYear,int *bLastDayMode,int *bLastDay,"
"int *b30Days,int *bUSMode,int *nDay);\n";

const char ScaDate[] =
"void ScaDate( int nNullDate, int nDate, int nBase,int *nOrigDay, "
"int *nMonth,int *nYear,int *bLastDayMode,int *bLastDay,"
"int *b30Days,int *bUSMode,int *nDay)\n"
"{\n"
"    DaysToDate( nNullDate + nDate, nOrigDay, nMonth, nYear );\n"
"    *bLastDayMode = (nBase != 5);\n"
"    *bLastDay = (*nOrigDay >= DaysInMonth( *nMonth, *nYear ));\n"
"    *b30Days = (nBase == 0) || (nBase == 4);\n"
"    *bUSMode = (nBase == 0);\n"
"    if( *b30Days)\n"
"    {\n"
"        *nDay = min( *nOrigDay, 30);\n"
"        if( *bLastDay || (*nDay >=DaysInMonth( *nMonth, *nYear )) )\n"
"        *nDay = 30;\n"
"    }\n"
"    else\n"
"    {\n"
"        int nLastDay = DaysInMonth( *nMonth, *nYear );\n"
"        *nDay = *bLastDay ? nLastDay : min( *nOrigDay, nLastDay );\n"
"    }\n"
"}\n";

const char lcl_GetCouppcdDecl[] =
"int lcl_GetCouppcd(int nNullDate,int nSettle,int nMat,int nFreq,int nBase);\n";

const char lcl_GetCouppcd[] =
"int lcl_GetCouppcd(int nNullDate,int nSettle, int nMat,int nFreq,int nBase)\n"
"{\n"
"    int aDate = nMat;\n"
"    int rDay=0,rMonth=0, rYear=0,rbLastDayMode=0, rbLastDay=0,rb30Days=0,"
"rbUSMode=0,rnDay=0;\n"
"    int sDay=0,sMonth=0, sYear=0,sbLastDayMode=0, sbLastDay=0,sb30Days=0,"
"sbUSMode=0,snDay=0;\n"
"    ScaDate( nNullDate,nSettle,nBase,&sDay,&sMonth,&sYear,&sbLastDayMode,"
"&sbLastDay,&sb30Days,&sbUSMode,&snDay);\n"
"    ScaDate( nNullDate,aDate,nBase,&rDay,&rMonth,&rYear,&rbLastDayMode,"
"&rbLastDay,&rb30Days,&rbUSMode,&rnDay);\n"
"    rYear=sYear;\n"
"    setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"    if(checklessthan(rYear,sYear,rMonth,sMonth,rnDay,snDay,rbLastDay,"
"sbLastDay,rDay,sDay))\n"
"    {\n"
"        rYear+=1;\n"
"        setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"    }\n"
"    while(checklessthan(sYear,rYear,sMonth,rMonth,snDay,rnDay,sbLastDay,"
"rbLastDay,sDay,rDay))\n"
"    {\n"
"        double d = -1*(12/nFreq);\n"
"        addMonths(rb30Days,rbLastDay,&rnDay,rDay,&rMonth,d,&rYear);\n"
"    }\n"
"    int nLastDay = DaysInMonth( rMonth, rYear );\n"
"    int nRealDay = (rbLastDayMode && rbLastDay) ? nLastDay :"
"min( nLastDay, rDay );\n"
"    return DateToDays( nRealDay, rMonth, rYear ) - nNullDate;\n"
"}\n";

const char lcl_GetCoupncdDecl[] =
"int lcl_GetCoupncd(int nNullDate,int nSettle,int nMat,int nFreq,int nBase);\n";

const char lcl_GetCoupncd[] =
"int lcl_GetCoupncd(int nNullDate,int nSettle, int nMat,int nFreq,int nBase)\n"
"{\n"
"    int aDate = nMat;\n"
"    int rDay=0,rMonth=0, rYear=0,rbLastDayMode=0, rbLastDay=0,rb30Days=0,"
"rbUSMode=0,rnDay=0;\n"
"    int sDay=0,sMonth=0, sYear=0,sbLastDayMode=0, sbLastDay=0,sb30Days=0,"
"sbUSMode=0,snDay=0;\n"
"    ScaDate( nNullDate,nSettle,nBase,&sDay,&sMonth,&sYear,&sbLastDayMode,"
"&sbLastDay,&sb30Days,&sbUSMode,&snDay);\n"
"    ScaDate( nNullDate,aDate,nBase,&rDay,&rMonth,&rYear,&rbLastDayMode,"
"&rbLastDay,&rb30Days,&rbUSMode,&rnDay);\n"
"    rYear=sYear;\n"
"    setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"    if(checklessthan(sYear,rYear,sMonth,rMonth,snDay,rnDay,sbLastDay,rbLastDay"
",sDay,rDay))\n"
"    {\n"
"        rYear-=1;\n"
"        setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"    }\n"
"    while(!checklessthan(sYear,rYear,sMonth,rMonth,snDay,rnDay,sbLastDay,"
"rbLastDay,sDay,rDay))\n"
"    {\n"
"      addMonths(rb30Days,rbLastDay,&rnDay,rDay,&rMonth,12/nFreq,&rYear);\n"
"    }\n"
"    int nLastDay = DaysInMonth( rMonth, rYear );\n"
"    int nRealDay = (rbLastDayMode && rbLastDay) ? nLastDay :"
"min( nLastDay, rDay );\n"
"    return DateToDays( nRealDay, rMonth, rYear ) - nNullDate;\n"
"}\n";

const char addMonthsDecl[] =
"void addMonths(int b30Days,int bLastDay,int *nDay,int nOrigDay,"
"int *nMonth,int nMonthCount,int *year);\n";

const char addMonths[] =
"void addMonths(int b30Days,int bLastDay,int *nDay,int nOrigDay,"
"int *nMonth,int nMonthCount,int *year)\n"
"{\n"
"    int nNewMonth = nMonthCount + *nMonth;\n"
"    if( nNewMonth > 12 )\n"
"    {\n"
"        --nNewMonth;\n"
"        *year+=nNewMonth / 12 ;\n"
"        *nMonth = ( nNewMonth % 12 ) + 1;\n"
"    }\n"
"    else if( nNewMonth < 1 )\n"
"    {\n"
"        *year+= nNewMonth / 12 - 1 ;\n"
"        *nMonth =  nNewMonth % 12 + 12 ;\n"
"    }\n"
"    else\n"
"        *nMonth = nNewMonth ;\n"
"    if( b30Days )\n"
"    {\n"
"        *nDay = min( nOrigDay, 30);\n"
"        if( bLastDay || (*nDay >= DaysInMonth( *nMonth, *year )) )\n"
"            *nDay = 30;\n"
"    }\n"
"    else\n"
"    {\n"
"        int nLastDay = DaysInMonth( *nMonth, *year );\n"
"        *nDay = bLastDay ? nLastDay : min( nOrigDay, nLastDay );\n"
"    }\n"
"}\n";

const char getDaysInMonthRangeDecl[] =
"int getDaysInMonthRange( int nFrom, int nTo,int b30Days,int year);\n";

const char getDaysInMonthRange[] =
"int getDaysInMonthRange( int nFrom, int nTo,int b30Days,int year)\n"
"{\n"
"    if( nFrom > nTo )\n"
"        return 0;\n"
"    int nRet = 0;\n"
"    if( b30Days )\n"
"        nRet = (nTo - nFrom + 1) * 30;\n"
"    else\n"
"    {\n"
"        for( int nMonthIx = nFrom; nMonthIx <= nTo; ++nMonthIx )\n"
"            nRet += b30Days ? 30 : DaysInMonth( nMonthIx, year );\n"
"    }\n"
"    return nRet;\n"
"}\n";

const char GetDaysInYearsDecl[] =
"int GetDaysInYears( int nYear1, int nYear2 );\n";

const char GetDaysInYears[] =
"int GetDaysInYears( int nYear1, int nYear2 )\n"
"{\n"
"    int  nLeaps = 0;\n"
"    for( int n = nYear1 ; n <= nYear2 ; n++ )\n"
"    {\n"
"        if( IsLeapYear( n ) )\n"
"            nLeaps++;\n"
"    }\n"
"    int  nSum = 1;\n"
"    nSum += nYear2;\n"
"    nSum -= nYear1;\n"
"    nSum *= 365;\n"
"    nSum += nLeaps;\n"
"    return nSum;\n"
"}\n";

const char GetDaysInYearDecl[] =
"int GetDaysInYear( int nNullDate, int nDate, int nMode );\n";

const char GetDaysInYear[] =
"int GetDaysInYear( int nNullDate, int nDate, int nMode )\n"
"{\n"
"    switch( nMode )\n"
"    {\n"
"    case 0:\n"
"    case 2:\n"
"    case 4:\n"
"        return 360;\n"
"    case 1:\n"
"        {\n"
"            int  nD=0, nM=0, nY=0;\n"
"            nDate += nNullDate;\n"
"            DaysToDate( nDate, &nD, &nM, &nY );\n"
"            return IsLeapYear( nY )? 366 : 365;\n"
"        }\n"
"    case 3:\n"
"        return 365;\n"
"    }\n"
"}\n";

const char getDaysInYearRangeDecl[] =
"int getDaysInYearRange( int nFrom, int nTo,int b30Days );\n";

const char getDaysInYearRange[] =
"int getDaysInYearRange( int nFrom, int nTo,int b30Days )\n"
"{\n"
"    if( nFrom > nTo )\n"
"        return 0;\n"
"    return b30Days ? ((nTo - nFrom + 1) * 360) : GetDaysInYears( nFrom, nTo)"
";\n"
"}\n";

const char getDiffDecl[] =
"int getDiff(int rFrom,int rTo,int fDay,int fMonth,int fYear,int fbLastDayMode,"
"int fbLastDay,int fb30Days,int fbUSMode,int fnDay,int tDay,int tMonth,"
"int tYear,int tbLastDayMode,int tbLastDay,int tb30Days,"
"int tbUSMode,int tnDay);\n";

const char getDiff[] =
"int getDiff(int rFrom,int rTo,int fDay,int fMonth,int fYear,int fbLastDayMode,"
"int fbLastDay,int fb30Days,int fbUSMode,int fnDay,int tDay,int tMonth,"
"int tYear,int tbLastDayMode,int tbLastDay,int tb30Days,"
"int tbUSMode,int tnDay)\n"
"{\n"
"    if(rFrom>rTo)\n"
"    {\n"
"        int d=fDay;fDay=tDay;tDay=d;\n"
"        int m=fMonth;fMonth=tMonth;tMonth=m;\n"
"        int y=fYear;fYear=tYear;tYear=y;\n"
"        int a=fbLastDayMode;fbLastDayMode=tbLastDayMode;tbLastDayMode=a;\n"
"        int b=fbLastDay;fbLastDay=tbLastDay;tbLastDay=b;\n"
"        int c=fb30Days;fb30Days=tb30Days;tb30Days=c;\n"
"        int e=fbUSMode;fbUSMode=tbUSMode;tbUSMode=e;\n"
"        int f=fnDay;fnDay=tnDay;tnDay=f;\n"
"    }\n"
"    int nDiff=0;\n"
"    if( tb30Days )\n"
"    {\n"
"        if( tbUSMode )\n"
"        {\n"
"            if( ((fMonth == 2) || (fnDay < 30)) && (tDay == 31) )\n"
"                tnDay = 31;\n"
"            else if( (tMonth == 2) && tbLastDay )\n"
"                tnDay = DaysInMonth( 2, tYear );\n"
"        }\n"
"        else\n"
"        {\n"
"            if( (fMonth == 2) && (fnDay == 30) )\n"
"                fnDay = DaysInMonth( 2, fYear );\n"
"            if( (tMonth == 2) && (tnDay == 30) )\n"
"                tnDay = DaysInMonth( 2, tYear );\n"
"        }\n"
"    }\n"
"    if( (fYear < tYear) || ((fYear == tYear) && (fMonth < tMonth)) )\n"
"    {\n"
"        int d = fb30Days ? 30:DaysInMonth(fMonth,fYear);\n"
"        nDiff = d- fnDay + 1;\n"
"        fDay = fnDay = 1;\n"
"        fbLastDay = 0;\n"
"        addMonths(fb30Days,fbLastDay,&fnDay,fDay,&fMonth,1,&fYear);\n"
"        if( fYear < tYear )\n"
"        {\n"
"            nDiff += getDaysInMonthRange( fMonth, 12,fb30Days,fYear);\n"
"            addMonths(fb30Days,fbLastDay,&fnDay,fDay,&fMonth,13-fMonth,&fYear"
");\n"
"            nDiff += getDaysInYearRange( fYear, tYear - 1,fb30Days);\n"
"            fYear+=tYear - fYear;\n"
"        }\n"
"        nDiff += getDaysInMonthRange(fMonth, tMonth - 1,fb30Days ,fYear );\n"
"        addMonths(fb30Days,fbLastDay,&fnDay,fDay,&fMonth,tMonth-fMonth,&fYear"
");\n"
"    }\n"
"    nDiff += tnDay - fnDay;\n"
"    return nDiff > 0 ? nDiff : 0;\n"
"}\n";

const char lcl_GetcoupdaybsDecl[] =
"int lcl_Getcoupdaybs(int nNullDate,int nSettle,int nMat,int nFreq,"
"int nBase);\n";

const char lcl_Getcoupdaybs[] =
"int lcl_Getcoupdaybs(int nNullDate,int nSettle,int nMat,int nFreq,"
"int nBase)\n"
"{\n"
"    int aDate = nMat;\n"
"    int rDay=0,rMonth=0, rYear=0,rbLastDayMode=0, rbLastDay=0,rb30Days=0,"
"rbUSMode=0,rnDay=0;\n"
"    int sDay=0,sMonth=0, sYear=0,sbLastDayMode=0, sbLastDay=0,sb30Days=0,"
"sbUSMode=0,snDay=0;\n"
"    ScaDate( nNullDate,nSettle,nBase,&sDay,&sMonth,&sYear,&sbLastDayMode,"
"&sbLastDay,&sb30Days,&sbUSMode,&snDay);\n"
"    ScaDate( nNullDate,aDate,nBase,&rDay,&rMonth,&rYear,&rbLastDayMode,"
"&rbLastDay,&rb30Days,&rbUSMode,&rnDay);\n"
"    rYear=sYear;\n"
"    setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"    aDate=DateToDays( rnDay,rMonth,rYear);\n"
"    if(checklessthan(rYear,sYear,rMonth,sMonth,rnDay,snDay,rbLastDay,"
"sbLastDay,rDay,sDay))\n"
"    {\n"
"        rYear+=1;\n"
"        setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"        aDate=DateToDays( rnDay,rMonth,rYear );\n"
"    }\n"
"    while(checklessthan(sYear,rYear,sMonth,rMonth,snDay,rnDay,sbLastDay,"
"rbLastDay,sDay,rDay))\n"
"    {\n"
"        double d = -1*(12/nFreq);\n"
"        addMonths(rb30Days,rbLastDay,&rnDay,rDay,&rMonth,d,&rYear);\n"
"        aDate=DateToDays( rnDay,rMonth,rYear );\n"
"    }\n"
"    return getDiff( aDate,nSettle+nNullDate,rDay,rMonth,rYear,rbLastDayMode,"
"rbLastDay,rb30Days,rbUSMode,rnDay,sDay,sMonth,sYear,sbLastDayMode,sbLastDay,"
"sb30Days,sbUSMode, snDay);\n"
"}\n";

const char lcl_GetcoupdaysDecl[] =
"int lcl_Getcoupdays(int nNullDate,int nSettle, "
"int nMat,int nFreq,int nBase);\n";

const char lcl_Getcoupdays[] =
"int lcl_Getcoupdays(int nNullDate,int nSettle, "
"int nMat,int nFreq,int nBase)\n"
"{\n"
"    int aDate = nMat;\n"
"    int rDay=0,rMonth=0, rYear=0,rbLastDayMode=0, rbLastDay=0,rb30Days=0,"
"rbUSMode=0,rnDay=0;\n"
"    int sDay=0,sMonth=0, sYear=0,sbLastDayMode=0, sbLastDay=0,sb30Days=0,"
"sbUSMode=0,snDay=0;\n"
"    ScaDate( nNullDate,nSettle,nBase,&sDay,&sMonth,&sYear,&sbLastDayMode,"
"&sbLastDay,&sb30Days,&sbUSMode,&snDay);\n"
"    ScaDate( nNullDate,aDate,nBase,&rDay,&rMonth,&rYear,&rbLastDayMode,"
"&rbLastDay,&rb30Days,&rbUSMode,&rnDay);\n"
"    rYear=sYear;\n"
"    setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"    aDate=DateToDays( rnDay,rMonth,rYear);\n"
"    if(checklessthan(rYear,sYear,rMonth,sMonth,rnDay,snDay,rbLastDay,"
"sbLastDay,rDay,sDay))\n"
"    {\n"
"        rYear+=1;\n"
"        setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"        aDate=DateToDays( rnDay,rMonth,rYear );\n"
"    }\n"
"    while(checklessthan(sYear,rYear,sMonth,rMonth,snDay,rnDay,sbLastDay,"
"rbLastDay,sDay,rDay))\n"
"    {\n"
"        double d = -1*12/(double)nFreq;\n"
"        addMonths(rb30Days,rbLastDay,&rnDay,rDay,&rMonth,d,&rYear);\n"
"        aDate=DateToDays( rnDay,rMonth,rYear );\n"
"    }\n"
"    int aNextDate=aDate;int aDay=rDay,aMonth=rMonth, aYear=rYear;\n"
"    int abLastDayMode=rbLastDayMode, abLastDay=rbLastDay,ab30Days=rb30Days,"
"abUSMode=rbUSMode,anDay=rnDay;\n"
"    int tmp = (int)(12/(double)nFreq);\n"
"    addMonths(ab30Days,abLastDay,&anDay,aDay,&aMonth,tmp,&aYear);\n"
"    return getDiff( aDate, aNextDate, rDay, rMonth, rYear, rbLastDayMode, "
"rbLastDay, rb30Days, rbUSMode, rnDay, aDay, aMonth, aYear, abLastDayMode,"
"abLastDay, ab30Days, abUSMode, anDay);\n"
"}\n";

const char lcl_GetcoupnumDecl[] =
"double lcl_Getcoupnum(int nNullDate,int nSettle,int nMat,int nFreq,int"
" nBase);\n";
const char lcl_Getcoupnum[] =
"double lcl_Getcoupnum(int nNullDate,int nSettle, int nMat,int nFreq,int"
" nBase)\n"
"{\n"
"    int aDate = nMat;\n"
"    int mDay=0,mMonth=0, mYear=0;\n"
"    int rDay=0,rMonth=0, rYear=0,rbLastDayMode=0, rbLastDay=0,rb30Days=0,"
"rbUSMode=0,rnDay=0;\n"
"    int sDay=0,sMonth=0, sYear=0,sbLastDayMode=0, sbLastDay=0,sb30Days=0,"
"sbUSMode=0,snDay=0;\n"
"    ScaDate( nNullDate,nSettle,nBase,&sDay,&sMonth,&sYear,&sbLastDayMode,"
"&sbLastDay,&sb30Days,&sbUSMode,&snDay);\n"
"    ScaDate( nNullDate,aDate,nBase,&rDay,&rMonth,&rYear,&rbLastDayMode,"
"&rbLastDay,&rb30Days,&rbUSMode,&rnDay);\n"
"    mMonth = rMonth, mYear = rYear;\n"
"    rYear=sYear;\n"
"    setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"    if(checklessthan(rYear,sYear,rMonth,sMonth,rnDay,snDay,rbLastDay,"
"sbLastDay,rDay,sDay))\n"
"    {\n"
"        rYear+=1;\n"
"        setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"    }\n"
"    int m= checklessthan(sYear,rYear,sMonth,rMonth,snDay,rnDay,sbLastDay,"
"rbLastDay,sDay,rDay);\n"
"    while(m)\n"
"    {\n"
"        double d = -1*(12/nFreq);\n"
"        addMonths(rb30Days,rbLastDay,&rnDay,rDay,&rMonth,d,&rYear);\n"
"        m = checklessthan(sYear,rYear,sMonth,rMonth,snDay,rnDay,sbLastDay,"
"rbLastDay,sDay,rDay);\n"
"    }\n"
"    int n=(mYear-rYear)*12+mMonth-rMonth;\n"
"    double tmp = (double)(n*nFreq)/12.0;\n"
"    return tmp;\n"
"}\n";

const char setDayDecl[] =
"void setDay(int nOrigDay, int nMonth,int nYear,int bLastDay,int b30Days,"
"int *nDay);\n";
const char setDay[] =
"void setDay(int nOrigDay, int nMonth,int nYear,int bLastDay,int b30Days,"
"int *nDay)\n"
"{\n"
"    if( b30Days )\n"
"    {\n"
"        *nDay = min( nOrigDay, 30);\n"
"        if( bLastDay || (*nDay >= DaysInMonth( nMonth, nYear )) )\n"
"            *nDay = 30;\n"
"    }\n"
"    else\n"
"    {\n"
"        int nLastDay = DaysInMonth( nMonth, nYear );\n"
"        *nDay = bLastDay ? nLastDay : min( nOrigDay, nLastDay );\n"
"    }\n"
"}\n";

const char coupdaysDecl[] =
"double coupdays(int nSettle,int nMat,int nFreq,int nBase);\n";

const char coupdays[] =
"double coupdays(int nSettle,int nMat,int nFreq,int nBase)\n"
"{\n"
"    int nNullDate=693594;\n"
"    if( nBase == 1 )\n"
"        return lcl_Getcoupdays(nNullDate, nSettle, nMat,nFreq, nBase);\n"
"    else\n"
"        return (double)GetDaysInYear(0,0,nBase)/(double)nFreq;\n"
"}\n";

const char coupdaybsDecl[] =
"double coupdaybs( int nSettle,int nMat,int nFreq,int nBase);\n";

const char coupdaybs[] =
"double coupdaybs( int nSettle,int nMat,int nFreq,int nBase)\n"
"{\n"
"    int nNullDate=693594;\n"
"    return lcl_Getcoupdaybs(nNullDate, nSettle, nMat,nFreq, nBase);\n"
"}\n";

const char coupdaysncDecl[] =
"double coupdaysnc( int nSettle,int nMat,int nFreq,int nBase);\n";

const char coupdaysnc[] =
"double coupdaysnc( int nSettle,int nMat,int nFreq,int nBase)\n"
"{\n"
"    int nNullDate=693594;\n"
"    if((nBase != 0) && (nBase != 4))\n"
"    {\n"
"    int aDate = nMat;\n"
"    int rDay=0,rMonth=0, rYear=0,rbLastDayMode=0, rbLastDay=0,rb30Days=0,"
"rbUSMode=0,rnDay=0;\n"
"    int sDay=0,sMonth=0, sYear=0,sbLastDayMode=0, sbLastDay=0,sb30Days=0,"
"sbUSMode=0,snDay=0;\n"
"    ScaDate( nNullDate,aDate,nBase,&rDay,&rMonth,&rYear,&rbLastDayMode,"
"&rbLastDay,&rb30Days,&rbUSMode,&rnDay);\n"
"    ScaDate( nNullDate,nSettle,nBase,&sDay,&sMonth,&sYear,&sbLastDayMode,"
"&sbLastDay,&sb30Days,&sbUSMode,&snDay);\n"
"    rYear=sYear;\n"
"    setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"    aDate=DateToDays( rnDay,rMonth,rYear);\n"
"    if(checklessthan(sYear,rYear,sMonth,rMonth,snDay,rnDay,sbLastDay,rbLastDay"
",sDay,rDay))\n"
"    {\n"
"        rYear-=1;\n"
"        setDay(rDay,rMonth,rYear,rbLastDay,rb30Days,&rnDay);\n"
"        aDate=DateToDays( rnDay,rMonth,rYear );\n"
"    }\n"
"    while(!checklessthan(sYear,rYear,sMonth,rMonth,snDay,rnDay,sbLastDay,"
"rbLastDay,sDay,rDay))\n"
"    {\n"
"      addMonths(rb30Days,rbLastDay,&rnDay,rDay,&rMonth,12/nFreq,&rYear);\n"
"      aDate=DateToDays( rnDay,rMonth,rYear );\n"
"    }\n"
"    return getDiff( nSettle+nNullDate,aDate,sDay,sMonth,sYear,sbLastDayMode, "
"sbLastDay, sb30Days, sbUSMode, snDay, rDay, rMonth, rYear, rbLastDayMode, "
"rbLastDay, rb30Days, rbUSMode, rnDay);\n"
"    }\n"
"    else\n"
"        return coupdays(nSettle,nMat,nFreq,nBase)- coupdaybs( nSettle,"
"nMat,nFreq,nBase);\n"
"}\n";

const char checklessthanDecl[] =
"int checklessthan(int aYear,int bYear,int aMonth,int bMonth,int anDay,int "
"bnDay,int abLastDay,int bbLastDay,int anOrigDay,int bnOrigDay);\n";
const char checklessthan[] =
"int checklessthan(int aYear,int bYear,int aMonth,int bMonth,int anDay,int "
"bnDay,int abLastDay,int bbLastDay,int anOrigDay,int bnOrigDay)\n"
"{\n"
"    if( aYear != bYear )\n"
"        return aYear < bYear;\n"
"    if( aMonth != bMonth )\n"
"        return aMonth < bMonth;\n"
"    if( anDay != bnDay )\n"
"        return anDay < bnDay;\n"
"    if( abLastDay || bbLastDay )\n"
"        return !abLastDay && bbLastDay;\n"
"    return anOrigDay < bnOrigDay;\n"
"}\n";

const char coupnumDecl[] =
"double coupnum( int nSettle,int nMat,int nFreq,int nBase);\n";

const char coupnum[] =
"double coupnum( int nSettle,int nMat,int nFreq,int nBase)\n"
"{\n"
"    int nNullDate=693594;\n"
"    return lcl_Getcoupnum(nNullDate,nSettle,nMat,nFreq,nBase);\n"
"}\n";

const char getPriceDecl[] =
"double getPrice(int nSettle, int nMat, double fRate, double fYield,\n"
    "double fRedemp, int nFreq, int nBase );\n";

const char getPrice[] =
"double getPrice(int nSettle, int nMat, double fRate, double fYield,\n"
    "double fRedemp, int nFreq, int nBase )\n"
"{\n"
"    double      fFreq = nFreq;\n"
"    double      fE = coupdays( nSettle, nMat, nFreq, nBase );\n"
"    double      fDSC_E = coupdaysnc(  nSettle, nMat, nFreq, nBase ) / fE;\n"
"    double      fN = coupnum( nSettle, nMat, nFreq, nBase );\n"
"    double      fA = coupdaybs( nSettle, nMat, nFreq, nBase );\n"
"    double      fRet = fRedemp / ( pow( 1.0 + fYield / fFreq, fN - 1.0 + "
"fDSC_E ) );\n"
"    fRet -= 100.0 * fRate / fFreq * fA / fE;\n"
"    double      fT1 = 100.0 * fRate / fFreq;\n"
"    double      fT2 = 1.0 + fYield / fFreq;\n"
"    for( double fK = 0.0 ; fK < fN ; fK+=1.0 )\n"
"        fRet += fT1 / pow( fT2, fK + fDSC_E );\n"
"    return fRet;\n"
"}\n";

const char getYield_Decl[] =
"double getYield_( int nNullDate, int nSettle, int nMat, double fCoup,"
"double fPrice,double fRedemp, int nFreq, int nBase);\n";

const char getYield_[] =
"double getYield_( int nNullDate, int nSettle, int nMat, double fCoup,"
"double fPrice,double fRedemp, int nFreq, int nBase )\n"
"{\n"
"    double      fRate = fCoup;\n"
"    double      fPriceN = 0.0;\n"
"    double      fYield1 = 0.0;\n"
"    double      fYield2 = 1.0;\n"
"    double      fPrice1 = getPrice(nSettle, nMat, fRate, fYield1, fRedemp, "
"nFreq, nBase );\n"
"    double      fPrice2 = getPrice(nSettle, nMat, fRate, fYield2, fRedemp, "
"nFreq, nBase );\n"
"    double      fYieldN = ( fYield2 - fYield1 ) * 0.5;\n"
"    for( unsigned int nIter = 0 ; nIter < 100 && fPriceN != fPrice ; nIter++ "
")\n"
"    {\n"
"        fPriceN = getPrice(nSettle, nMat, fRate, fYieldN, fRedemp, nFreq, "
"nBase );\n"
"        if( fPrice == fPrice1 )\n"
"            return fYield1;\n"
"        else if( fPrice == fPrice2 )\n"
"            return fYield2;\n"
"        else if( fPrice == fPriceN )\n"
"            return fYieldN;\n"
"        else if( fPrice < fPrice2 )\n"
"        {\n"
"            fYield2 *= 2.0;\n"
"            fPrice2 = getPrice(nSettle, nMat, fRate, fYield2, fRedemp, nFreq"
", nBase );\n"
"            fYieldN = ( fYield2 - fYield1 ) * 0.5;\n"
"        }\n"
"        else\n"
"        {\n"
"            if( fPrice < fPriceN )\n"
"            {\n"
"                fYield1 = fYieldN;\n"
"                fPrice1 = fPriceN;\n"
"            }\n"
"            else\n"
"            {\n"
"                fYield2 = fYieldN;\n"
"                fPrice2 = fPriceN;\n"
"            }\n"
"            fYieldN = fYield2 - ( fYield2 - fYield1 ) * ( ( fPrice - fPrice2 "
") / ( fPrice1 - fPrice2 ) );\n"
"        }\n"
"    }\n"
"    return fYieldN;\n"
"}\n";

const char GetYieldmatDecl[] =
    "double GetYieldmat( int nNullDate, int nSettle, int nMat, int nIssue,\n"
        "double fRate, double fPrice, int nBase );\n";

const char GetYieldmat[] =
    "double GetYieldmat( int nNullDate, int nSettle, int nMat, int nIssue,\n"
        "double fRate, double fPrice, int nBase )\n"
"{\n"
"    double      fIssMat = GetYearFrac( nNullDate, nIssue, nMat, nBase );\n"
"    double      fIssSet = GetYearFrac( nNullDate, nIssue, nSettle, nBase );\n"
"    double      fSetMat = GetYearFrac( nNullDate, nSettle, nMat, nBase );\n"
"    double      y = 1.0 + fIssMat * fRate;\n"
"    y =y / (fPrice / 100.0 + fIssSet * fRate);\n"
"    y-=1.0;\n"
"    y = y / fSetMat;\n"
"    return y;\n"
"}\n";

const char GetDiffDate360_Decl[] =
"int GetDiffDate360_(\n"
"                int nDay1, int nMonth1, int nYear1, bool bLeapYear1,\n"
"                int nDay2, int nMonth2, int nYear2,\n"
"                bool bUSAMethod );\n";

const char GetDiffDate360_[] =
"int GetDiffDate360_(\n"
"                int nDay1, int nMonth1, int nYear1, bool bLeapYear1,\n"
"                int nDay2, int nMonth2, int nYear2,\n"
"                bool bUSAMethod )\n"
"{\n"
"    if( nDay1 == 31 )\n"
"        nDay1--;\n"
"    else if( bUSAMethod && ( nMonth1 == 2 && ( nDay1 == 29 || ( nDay1 == 28 "
"&& !bLeapYear1 ) ) ) )\n"
"        nDay1 = 30;\n"
"    if( nDay2 == 31 )\n"
"    {\n"
"        if( bUSAMethod && nDay1 != 30 )\n"
"        {\n"
"            nDay2 = 1;\n"
"            if( nMonth2 == 12 )\n"
"            {\n"
"                nYear2++;\n"
"                nMonth2 = 1;\n"
"            }\n"
"            else\n"
"                nMonth2++;\n"
"        }\n"
"        else\n"
"            nDay2 = 30;\n"
"    }\n"
"    return nDay2 + nMonth2 * 30 + nYear2 * 360 - nDay1 - nMonth1 * 30 - "
"nYear1 * 360;\n"
"}\n";

const char GetDiffDate360Decl[] =
"int GetDiffDate360( int nNullDate, int nDate1, int nDate2,"
"bool bUSAMethod);\n";

const char GetDiffDate360[] =
"int GetDiffDate360( int nNullDate, int nDate1, int nDate2,"
"bool bUSAMethod )\n"
"{\n"
"    nDate1 += nNullDate;\n"
"    nDate2 += nNullDate;\n"
"    int nDay1, nMonth1, nYear1, nDay2, nMonth2, nYear2;\n"
"    DaysToDate( nDate1, &nDay1, &nMonth1, &nYear1 );\n"
"    DaysToDate( nDate2, &nDay2, &nMonth2, &nYear2 );\n"
"    return GetDiffDate360_( nDay1, nMonth1, nYear1, IsLeapYear( nYear1 ), "
"nDay2, nMonth2, nYear2, bUSAMethod );\n"
"}\n";

const char GetDurationDecl[] =
"double GetDuration( \n"
"                int nNullDate, int nSettle, int nMat, double fCoup,\n"
"                double fYield, int nFreq, int nBase );\n";

const char GetDuration[] =
"double GetDuration( \n"
"                int nNullDate, int nSettle, int nMat, double fCoup,\n"
"                double fYield, int nFreq, int nBase )\n"
"    {\n"
"        double fYearfrac = GetYearFrac(nNullDate,nSettle,nMat,nBase);\n"
"        double fNumOfCoups = lcl_Getcoupnum(nNullDate,nSettle,nMat,"
"nFreq,nBase);\n"
"        double fDur = 0.0;\n"
"        fCoup = fCoup * 100.0 / nFreq;\n"
"        fYield = fYield / nFreq;\n"
"        fYield += 1.0;\n"
"        double nDiff = fYearfrac * nFreq - fNumOfCoups;\n"
"        int  t;\n"
"        double tmp0 = 0, tmp1 = 0, tmp2 = 0;\n"
"        for( t = 1 ; t < fNumOfCoups ; t++ ){\n"
"            tmp0 = (t + nDiff) * ( fCoup ) ;\n"
"            tmp1 = pow( fYield, t + nDiff ) ;\n"
"            tmp2 = tmp0 / tmp1;\n"
"            fDur += tmp2;\n"
"        }\n"
"        fDur += (fNumOfCoups + nDiff) * (fCoup + 100.0) * pow(pow(fYield,"
" fNumOfCoups + nDiff ),-1);\n"
"        double  p = 0.0;\n"
"        for( t = 1 ; t < fNumOfCoups ; t++ ){\n"
"            tmp0 = pow( fYield, t + nDiff );\n"
"            p += fCoup / tmp0;}\n"
"        p += (fCoup + 100.0) / pow(fYield, fNumOfCoups + nDiff);\n"
"        fDur = fDur / p;\n"
"        fDur = fDur / nFreq;\n"
"        return fDur;\n"
"    }\n";

const char ScGetDDBDecl[] =
"double ScGetDDB(double fCost, double fSalvage, double fLife, double fPeriod,"
"double fFactor);\n";

const char ScGetDDB[] =
"double ScGetDDB(double fCost, double fSalvage, double fLife, double fPeriod,"
"double fFactor)\n"
"{\n"
"    double fDdb, fRate, fOldValue, fNewValue;\n"
"    fRate = fFactor / fLife;\n"
"    if (fRate >= 1.0)\n"
"    {\n"
"        fRate = 1.0;\n"
"        if (fPeriod == 1.0)\n"
"            fOldValue = fCost;\n"
"        else\n"
"            fOldValue = 0.0;\n"
"    }\n"
"    else\n"
"        fOldValue = fCost * pow(1.0 - fRate, fPeriod - 1.0);\n"
"    fNewValue = fCost * pow(1.0 - fRate, fPeriod);\n"

"    if (fNewValue < fSalvage)\n"
"        fDdb = fOldValue - fSalvage;\n"
"    else\n"
"        fDdb = fOldValue - fNewValue;\n"
"    if (fDdb < 0.0)\n"
"        fDdb = 0.0;\n"
"    return fDdb;\n"
"}\n";

const char ScInterVDBDecl[] =
"double ScInterVDB(double fCost, double fSalvage, double fLife, double fLife1,"
"double fPeriod, double fFactor);\n";

const char ScInterVDB[] =
"double ScInterVDB(double fCost, double fSalvage, double fLife, double fLife1,"
"double fPeriod, double fFactor)\n"
"{\n"
"    double fVdb=0;\n"
"    double fIntEnd   = ceil(fPeriod);\n"
"    int nLoopEnd   = fIntEnd;\n"

"    double fTerm, fSln;\n"
"    double fSalvageValue = fCost - fSalvage;\n"
"    int nNowSln = 0;\n"
"    double fDdb;\n"
"    int i;\n"
"    fSln=0;\n"
"    for ( i = 1; i <= nLoopEnd; i++)\n"
"    {\n"
"        if(!nNowSln)\n"
"        {\n"
"            fDdb = ScGetDDB(fCost, fSalvage, fLife, (double) i, fFactor);\n"
"            fSln = fSalvageValue/ (fLife1 - (double) (i-1));\n"
"            if (fSln > fDdb)\n"
"            {\n"
"                fTerm = fSln;\n"
"                nNowSln = 1;\n"
"            }\n"
"            else\n"
"            {\n"
"                fTerm = fDdb;\n"
"                fSalvageValue =fSalvageValue- fDdb;\n"
"            }\n"
"        }\n"
"        else\n"
"        {\n"
"            fTerm = fSln;\n"
"        }\n"

"        if ( i == nLoopEnd)\n"
"            fTerm *= ( fPeriod + 1.0 - fIntEnd );\n"

"        fVdb += fTerm;\n"
"    }\n"
"    return fVdb;\n"
"}\n";

const char VDBImplementDecl[] =
"double VDBImplement(double fCost, double fSalvage, double fLife, double fStart"
", double fEnd, double fFactor, bool bNoSwitch);\n";

const char VDBImplement[] =
"double VDBImplement(double fCost, double fSalvage, double fLife, double fStart"
", double fEnd, double fFactor, bool bNoSwitch)\n"
"{\n"
"    double fIntStart = floor(fStart);\n"
"    double fIntEnd   = ceil(fEnd);\n"
"    long nLoopStart = (long)(fIntStart);\n"
"    long nLoopEnd   = (long)(fIntEnd);\n"
"\n"
"    double fVdb = 0.0;\n"
"    if (bNoSwitch)\n"
"    {\n"
"        for (long i = nLoopStart + 1; i <= nLoopEnd; i++)\n"
"        {\n"
"            double fTerm = ScGetDDB(fCost, fSalvage, fLife, i, fFactor);\n"
"\n"
"            if ( i == nLoopStart+1 )\n"
"                fTerm *= ( min( fEnd, fIntStart + 1.0 ) - fStart );\n"
"            else if ( i == nLoopEnd )\n"
"                fTerm *= ( fEnd + 1.0 - fIntEnd );\n"
"\n"
"            fVdb += fTerm;\n"
"        }\n"
"    }\n"
"    else\n"
"    {\n"
"        double fPart = 0.0;\n"
"        if ( !approx_equal( fStart, fIntStart ) ||\n"
"             !approx_equal( fEnd, fIntEnd ) )\n"
"        {\n"
"            if ( !approx_equal( fStart, fIntStart ) )\n"
"            {\n"
"                double fTempIntEnd = fIntStart + 1.0;\n"
"                double fTempValue = fCost -\n"
"                    ScInterVDB( fCost, fSalvage, fLife, fLife, fIntStart, fFactor );\n"
"                fPart += ( fStart - fIntStart ) *\n"
"                    ScInterVDB( fTempValue, fSalvage, fLife, fLife - fIntStart,\n"
"                    fTempIntEnd - fIntStart, fFactor);\n"
"            }\n"
"            if ( !approx_equal( fEnd, fIntEnd ) )\n"
"            {\n"
"                double fTempIntStart = fIntEnd - 1.0;\n"
"                double fTempValue = fCost -\n"
"                    ScInterVDB( fCost, fSalvage, fLife, fLife, fTempIntStart, fFactor );\n"
"                fPart += ( fIntEnd - fEnd ) *\n"
"                    ScInterVDB( fTempValue, fSalvage, fLife, fLife - fTempIntStart,\n"
"                    fIntEnd - fTempIntStart, fFactor);\n"
"            }\n"
"        }\n"
"        fCost -= ScInterVDB( fCost, fSalvage, fLife, fLife, fIntStart, fFactor );\n"
"        fVdb = ScInterVDB( fCost, fSalvage, fLife, fLife - fIntStart,\n"
"            fIntEnd - fIntStart, fFactor);\n"
"        fVdb -= fPart;\n"
"    }\n"
"    return fVdb;\n"
"}\n";

const char GetOddlpriceDecl[] =
"double GetOddlprice( int nNullDate, int nSettle, int nMat, int nLastCoup,\n"
"    double fRate, double fYield, double fRedemp, int nFreq, int nBase );\n";

const char GetOddlprice[] =
"double GetOddlprice( int nNullDate, int nSettle, int nMat, int nLastCoup,\n"
"    double fRate, double fYield, double fRedemp, int nFreq, int nBase )\n"
"{\n"
"    double   fFreq =  nFreq ;\n"
"    double   fDCi = GetYearFrac( nNullDate, nLastCoup,"
"nMat, nBase ) * fFreq;\n"
"    double   fDSCi = GetYearFrac( nNullDate, nSettle,"
"nMat, nBase ) * fFreq;\n"
"    double   fAi = GetYearFrac( nNullDate, nLastCoup,"
"nSettle, nBase ) * fFreq;\n"
"    double   p = fRedemp + fDCi * 100.0 * fRate / fFreq;\n"
"    p /= fDSCi * fYield / fFreq + 1.0;\n"
"    p -= fAi * 100.0 * fRate / fFreq;\n"
"    return p;\n"
"}\n";

const char GetOddlyieldDecl[] =
"double GetOddlyield( int nNullDate, int nSettle, int nMat, int nLastCoup,\n"
"    double fRate, double fPrice, double fRedemp, int nFreq, int nBase );\n";

const char GetOddlyield[] =
"double GetOddlyield( int nNullDate, int nSettle, int nMat, int nLastCoup,\n"
"    double fRate, double fPrice, double fRedemp, int nFreq, int nBase ) \n"
"{\n"
"    double fFreq =  nFreq ;\n"
"    double fDCi= GetYearFrac( nNullDate, nLastCoup, nMat, nBase ) * fFreq;\n"
"    double fDSCi= GetYearFrac( nNullDate, nSettle, nMat, nBase ) * fFreq;\n"
"    double fAi= GetYearFrac( nNullDate, nLastCoup, nSettle, nBase )*fFreq;\n"
"    double y = fRedemp + fDCi * 100.0 * fRate / fFreq;\n"
"    y /= fPrice + fAi * 100.0 * fRate / fFreq;\n"
"    y -= 1.0;\n"
"    y *= fFreq / fDSCi;\n"
"    return y;\n"
"}\n";

const char GetYearFracDecl[] =
"double GetYearFrac( int nNullDate, int nStartDate, int nEndDate,"
"int nMode );\n";

const char GetYearFrac[] =
"double GetYearFrac( int nNullDate, int nStartDate, int nEndDate,"
"int nMode ) \n"
"{\n"
"    if( nStartDate == nEndDate )\n"
"        return 0.0;     \n"
"   if( nStartDate > nEndDate )\n"
"   {\n"
"       int   n = nEndDate;\n"
"       nEndDate = nStartDate;\n"
"       nStartDate = n;\n"
"   }\n"
"  int nDate1 = nStartDate + nNullDate;\n"
"  int nDate2 = nEndDate + nNullDate;\n"
"  int  nDay1, nDay2;\n"
"  int  nMonth1, nMonth2;\n"
"  int  nYear1, nYear2;\n"
"  DaysToDate( nDate1, &nDay1, &nMonth1, &nYear1 );\n"
"  DaysToDate( nDate2, &nDay2, &nMonth2, &nYear2 );\n"
"  int nDayDiff;\n"
"  switch( nMode )\n"
"  {\n"
"      case 0:         \n"
"          if ( nDay1 == 31 )\n"
"          {\n"
"              nDay1--;\n"
"          }\n"
"          if ( nDay1 == 30 && nDay2 == 31 )\n"
"          {\n"
"              nDay2--;\n"
"          }\n"
"          else\n"
"          {\n"
"              if ( nMonth1 == 2 && nDay1 == "
"( IsLeapYear( nYear1 ) ? 29 : 28 ) )\n"
"              {\n"
"                  nDay1 = 30;\n"
"                  if ( nMonth2 == 2 && nDay2 == "
"( IsLeapYear( nYear2 ) ? 29 : 28 ) )\n"
"                  {\n"
"                      nDay2 = 30;\n"
"                  }\n"
"              }\n"
"          }\n"
"          nDayDiff = ( nYear2 - nYear1 ) * 360 + "
"( nMonth2 - nMonth1 ) * 30 + ( nDay2 - nDay1 );\n"
"          break;\n"
"      case 1:         \n"
"      case 2:        \n"
"      case 3:         \n"
"         nDayDiff = nDate2 - nDate1;\n"
"          break;\n"
"      case 4:         \n"
"          if ( nDay1 == 31 )\n"
"          {\n"
"              nDay1--;\n"
"          }\n"
"          if ( nDay2 == 31 )\n"
"          {\n"
"              nDay2--;\n"
"          }\n"
"          nDayDiff = ( nYear2 - nYear1 ) * 360 + "
"( nMonth2 - nMonth1 ) * 30 + ( nDay2 - nDay1 );\n"
"          break;\n"
"  }\n"
"  double nDaysInYear;\n"
"  switch( nMode )\n"
"  {\n"
"      case 0:         \n"
"      case 2:         \n"
"      case 4:         \n"
"          nDaysInYear = 360;\n"
"          break;\n"
"      case 1:         \n"
"          {\n"
"              bool isYearDifferent = ( nYear1 != nYear2 );\n"
"              if ( isYearDifferent &&\n"
"                   ( ( nYear2 != nYear1 + 1 ) ||\n"
"                     ( nMonth1 < nMonth2 ) ||\n"
"                     ( nMonth1 == nMonth2 && nDay1 < nDay2 ) ) )\n"
"              {\n"
"                  int nDayCount = 0;\n"
"                 for ( int i = nYear1; i <= nYear2; i++ )\n"
"                      nDayCount += ( IsLeapYear( i ) ? 366 : 365 );\n"
"                  nDaysInYear = ( double ) nDayCount / "
"( double ) ( nYear2 - nYear1 + 1 );\n"
"              }\n"
"             else\n"
"              {\n"
"                  if ( isYearDifferent && IsLeapYear( nYear1 ) )\n"
"                  {\n"
"                      nDaysInYear = 366;\n"
"                  }\n"
"                   else\n"
"                  {\n"
"                      if ( ( IsLeapYear( nYear1 ) && nMonth1 <= 2 "
"&& nDay1 <= 29 ) ||\n"
"                           ( IsLeapYear( nYear2 ) && ( nMonth2 > 3 || "
"( nMonth2 == 2 && nDay1 == 29 ) ) ) )\n"
"                     {\n"
"                         nDaysInYear = 366;\n"
"                     }\n"
"                      else\n"
"                      {\n"
"                          nDaysInYear = 365;\n"
"                           for ( int i = nYear1; i <= nYear2; i++ )\n"
"                          {\n"
"                             if ( IsLeapYear( i ) )\n"
"                              {\n"
"                                  nDaysInYear = 366;\n"
"                                 break;\n"
"                             }\n"
"                          }\n"
"                     }\n"
"                 }\n"
"             }\n"
"         }\n"
"          break;\n"
"      case 3:         \n"
"          nDaysInYear = 365;\n"
"           break;\n"
"  }\n"
"  return (double)( nDayDiff ) / (nDaysInYear);\n"
"}\n";

const char DaysToDateDecl[] =
"void DaysToDate( int nDays, int *rDay, int* rMonth, int* rYear );\n";

const char DaysToDate[] =
"void DaysToDate( int nDays, int *rDay, int* rMonth, int* rYear )\n"
"{\n"
"    int   nTempDays;\n"
"    int   i = 0;\n"
"    bool    bCalc;\n"
"    do\n"
"    {\n"
"        nTempDays = nDays;\n"
"        *rYear = (int)((nTempDays / 365) - i);\n"
"        nTempDays -= ((int) *rYear -1) * 365;\n"
"        nTempDays -= ((*rYear -1) / 4) - ((*rYear -1) / 100) + ((*rYear -1)"
" / 400);\n"
"        bCalc = false;\n"
"        if ( nTempDays < 1 )\n"
"        {\n"
"            i++;\n"
"            bCalc = true;\n"
"        }\n"
"        else\n"
"        {\n"
"            if ( nTempDays > 365 )\n"
"            {\n"
"                if ( (nTempDays != 366) || !IsLeapYear( *rYear ) )\n"
"                {\n"
"                    i--;\n"
"                    bCalc = true;\n"
"                }\n"
"            }\n"
"        }\n"
"    }\n"
"    while ( bCalc );\n"
"    if(nTempDays != 0){\n"
"        for (*rMonth = 1; (int)nTempDays > DaysInMonth( *rMonth, *rYear );"
" *rMonth += 1)\n"
"        {\n"
"            nTempDays -= DaysInMonth( *rMonth, *rYear ); \n"
"        }\n"
"        *rDay = (int)nTempDays;\n"
"    }\n"
"}\n";

const char DaysToDate_LocalBarrierDecl[] =
"void DaysToDate( int nDays, int *rDay, int* rMonth, int* rYear );\n";

const char DaysToDate_LocalBarrier[] =
"void DaysToDate( int nDays, int *rDay, int* rMonth, int* rYear )\n"
"{\n"
"    int   nTempDays;\n"
"    int   i = 0;\n"
"    bool    bCalc;\n"
"    do\n"
"    {\n"
"        nTempDays = nDays;\n"
"        *rYear = (int)((nTempDays / 365) - i);\n"
"        nTempDays -= ((int) *rYear -1) * 365;\n"
"        nTempDays -= ((*rYear -1) / 4) - ((*rYear -1) / 100) + ((*rYear -1)"
" / 400);\n"
"        bCalc = false;\n"
"        if ( nTempDays < 1 )\n"
"        {\n"
"            i++;\n"
"            bCalc = true;\n"
"        }\n"
"        else\n"
"        {\n"
"            if ( nTempDays > 365 )\n"
"            {\n"
"                if ( (nTempDays != 366) || !IsLeapYear( *rYear ) )\n"
"                {\n"
"                    i--;\n"
"                    bCalc = true;\n"
"                }\n"
"            }\n"
"        }\n"
"    }\n"
"    while ( bCalc );\n"
"    barrier(CLK_LOCAL_MEM_FENCE);\n"
"    if(nTempDays != 0){\n"
"        for (*rMonth = 1; (int)nTempDays > DaysInMonth( *rMonth, *rYear );"
" *rMonth += 1)\n"
"        {\n"
"            nTempDays -= DaysInMonth( *rMonth, *rYear ); \n"
"        }\n"
"        *rDay = (int)nTempDays;\n"
"    }\n"
"}\n";

const char GetYearDiffDecl[] =
"double GetYearDiff( int nNullDate, int nStartDate, int nEndDate,"
"int nMode);\n";

const char GetYearDiff[] =
"double GetYearDiff( int nNullDate, int nStartDate, int nEndDate,"
"int nMode )\n"
"{\n"
"    int   nDays1stYear;\n"
"    int   nTotalDays = GetDiffDate( nNullDate, nStartDate, nEndDate,"
"nMode, &"
"nDays1stYear );\n"
"    return (double)(nTotalDays) / (double)nDays1stYear;\n"
"}\n";

const char GetDiffDateDecl[] =
"int GetDiffDate( int nNullDate, int nStartDate, int nEndDate, int nMode,"
"    int* pOptDaysIn1stYear );\n";

const char GetDiffDate[] =
"int GetDiffDate( int nNullDate, int nStartDate, int nEndDate, int nMode,"
"    int* pOptDaysIn1stYear )\n"
"{\n"
"    bool    bNeg = nStartDate > nEndDate;\n"
"    if( bNeg )\n"
"    {\n"
"        int   n = nEndDate;\n"
"        nEndDate = nStartDate;\n"
"        nStartDate = n;\n"
"    }\n"
"    int       nRet;\n"
"    switch( nMode )\n"
"    {\n"
"    case 0:   \n"
"    case 4:   \n"
"        {\n"
"        int      nD1, nM1, nY1, nD2, nM2, nY2;\n"
"        nStartDate += nNullDate;\n"
"        nEndDate += nNullDate;\n"
"        DaysToDate( nStartDate, &nD1, &nM1, &nY1 );\n"
"        DaysToDate( nEndDate, &nD2, &nM2, &nY2 );\n"
"        bool        bLeap = IsLeapYear( nY1 );\n"
"        int       nDays, nMonths;\n"
"        nMonths = nM2 - nM1;\n"
"        nDays = nD2 - nD1;\n"
"        nMonths += ( nY2 - nY1 ) * 12;\n"
"        nRet = nMonths * 30 + nDays;\n"
"        if( nMode == 0 && nM1 == 2 && nM2 != 2 && nY1 == nY2 )\n"
"            nRet -= bLeap? 1 : 2;\n"
"        if( pOptDaysIn1stYear )\n"
"            *pOptDaysIn1stYear = 360;\n"
"        }\n"
"        break;\n"
"    case 1:    \n"
"        if( pOptDaysIn1stYear )\n"
"        {\n"
"            int      nD, nM, nY;\n"
"            DaysToDate( nStartDate + nNullDate, &nD, &nM, &nY );\n"
"            *pOptDaysIn1stYear = IsLeapYear( nY )? 366 : 365;\n"
"        }\n"
"        nRet = nEndDate - nStartDate;\n"
"        break;\n"
"    case 2:      \n"
"        nRet = nEndDate - nStartDate;\n"
"        if( pOptDaysIn1stYear )\n"
"            *pOptDaysIn1stYear = 360;\n"
"        break;\n"
"    case 3:        \n"
"        nRet = nEndDate - nStartDate;\n"
"        if( pOptDaysIn1stYear )\n"
"            *pOptDaysIn1stYear = 365;\n"
"        break;\n"
"    }\n"
"    return bNeg? -nRet : nRet;\n"
"}\n";

const char RateIterationDecl[] =
"bool RateIteration( double fNper, double fPayment, double fPv,"
"                    double fFv, bool bPayType, double* fGuess );\n";

const char RateIteration[] =
"bool RateIteration( double fNper, double fPayment, double fPv,"
"                    double fFv, bool bPayType, double* fGuess )\n"
"{\n"
"#define SCdEpsilon                1.0E-7\n"
"    bool bValid = true, bFound = false;\n"
"    double fX, fXnew, fTerm, fTermDerivation;\n"
"    double fGeoSeries, fGeoSeriesDerivation;\n"
"    const int nIterationsMax = 150;\n"
"    int nCount = 0;\n"
"    const double fEpsilonSmall = 1.0E-14;\n"
"    if ( bPayType )\n"
"    {\n"
"        fFv = fFv - fPayment;\n"
"        fPv = fPv + fPayment;\n"
"    }\n"
"    if (fNper == round( fNper ))\n"
"    {\n"
"        fX = *fGuess;\n"
"        while (!bFound && nCount < nIterationsMax)\n"
"        {\n"
"            double fPowN, fPowNminus1;\n"
"            fPowNminus1 = pow( 1.0+fX, fNper-1.0);\n"
"            fPowN = fPowNminus1 * (1.0+fX);\n"
"            if (fX == 0.0)\n"
"            {\n"
"                fGeoSeries = fNper;\n"
"                fGeoSeriesDerivation = fNper * (fNper-1.0)/2.0;\n"
"            }\n"
"            else\n"
"            {\n"
"                fGeoSeries = (fPowN-1.0)/fX;\n"
"                fGeoSeriesDerivation = fNper * fPowNminus1 / fX - fGeoSeries / fX;\n"
"            }\n"
"            fTerm = fFv + fPv *fPowN+ fPayment * fGeoSeries;\n"
"            fTermDerivation = fPv * fNper * fPowNminus1 + fPayment * fGeoSeriesDerivation;\n"
"            if (fabs(fTerm) < fEpsilonSmall)\n"
"                bFound = true;\n"
"            else\n"
"            {\n"
"                if (fTermDerivation == 0.0)\n"
"                    fXnew = fX + 1.1 * SCdEpsilon;\n"
"                else\n"
"                    fXnew = fX - fTerm / fTermDerivation;\n"
"                nCount++;\n"
"                bFound = (fabs(fXnew - fX) < SCdEpsilon);\n"
"                fX = fXnew;\n"
"            }\n"
"        }\n"
"        bValid = (fX > -1.0);\n"
"    }\n"
"    else\n"
"    {\n"
"        fX = (*fGuess < -1.0) ? -1.0 : *fGuess;\n"
"        while (bValid && !bFound && nCount < nIterationsMax)\n"
"        {\n"
"            if (fX == 0.0)\n"
"            {\n"
"                fGeoSeries = fNper;\n"
"                fGeoSeriesDerivation = fNper * (fNper-1.0)/2.0;\n"
"            }\n"
"            else\n"
"            {\n"
"                fGeoSeries = (pow( 1.0+fX, fNper) - 1.0) / fX;\n"
"                fGeoSeriesDerivation = fNper * pow( 1.0+fX, fNper-1.0) / fX - fGeoSeries / fX;\n"
"            }\n"
"            fTerm = fFv + fPv *pow(1.0 + fX,fNper)+ fPayment * fGeoSeries;\n"
"            fTermDerivation = fPv * fNper * pow( 1.0+fX, fNper-1.0) + fPayment * fGeoSeriesDerivation;\n"
"            if (fabs(fTerm) < fEpsilonSmall)\n"
"                bFound = true;\n"
"            else\n"
"            {\n"
"                if (fTermDerivation == 0.0)\n"
"                    fXnew = fX + 1.1 * SCdEpsilon;\n"
"                else\n"
"                    fXnew = fX - fTerm / fTermDerivation;\n"
"                nCount++;\n"
"                bFound = (fabs(fXnew - fX) < SCdEpsilon);\n"
"                fX = fXnew;\n"
"                bValid = (fX >= -1.0);\n"
"            }\n"
"        }\n"
"    }\n"
"    *fGuess = fX;\n"
"    return bValid && bFound;\n"
"}\n";

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
